;;;
;;; Hilfsmittel
;;;

;; Wir verwenden das Aussagenlogik-Modul und "uberschreiben
;; einige seiner Funktionen.

(load "logic.scm")

;;;
;;; Interne Repr"asentation von Pr"adikatenlogik
;;;

;; Konstruktoren

(define (make-function-symbol name)
  (cons 'function name))

(define (make-predicate-symbol name)
  (cons 'predicate name))

(define (make-variable name)
  (cons 'variable name))
;(define (make-variable name) name)  

(define (make-function-application name term-list)
  (cons (make-function-symbol name) term-list))

(define (make-constant name)
  (make-function-application name '()))

(define (make-predicate-application name term-list)
  (cons (make-predicate-symbol name) term-list))

(define (make-forall-quantification variable formula)
  (list 'forall variable formula))

(define (make-exists-quantification variable formula)
  (list 'exists variable formula))

;; Pr"adikate

(define (function-symbol? x)
  (and (pair? x)
       (equal? (car x) 'function)))

(define (predicate-symbol? x)  
  (and (pair? x)
       (equal? (car x) 'predicate)))

(define (variable? x)  
  (and (pair? x)
       (equal? (car x) 'variable)))
;(define variable? symbol?)

(define (forall-quantification? formula)
  (and (pair? formula)
       (equal? (car formula) 'forall)))

(define (exists-quantification? formula)
  (and (pair? formula)
       (equal? (car formula) 'exists)))

(define (function-application? formula)
  (and (pair? formula)
       (function-symbol? (car formula))))

(define (predicate-application? formula)
  (and (pair? formula)
       (predicate-symbol? (car formula))))

;; Selektor f"ur den Namen eines Funktionssymbols 
;; oder Pr"adikatssymbols

(define name cdr)

;; Selektor f"ur den Namen einer Variablen

(define (variable-name term) (cdr term))
;(define (variable-name term) term)

;; Selektoren f"ur Funktions- oder Pr"adikats"=Anwendungen

(define applied-symbol car)
(define (applied-symbol-name application) 
  (name (applied-symbol application)))
(define term-list cdr)

(define (arity application)
  ;; Ermittelt die Stelligkeit einer Pr"adikats- 
  ;; oder Funktions"=Anwendung.
  (length (term-list application)))
  
;; Selektoren f"ur quantifizierte Ausdr"ucke

(define quantified-variable cadr)
(define subject-to-quantification caddr)

;;; 
;;; Hilfsmittel f"ur externe Repr"asentationen
;;;

(define (string-upcase string)
  (list->string (map char-upcase (string->list string))))

(define (string-capitalize string)
  (let ((char-list (string->list string)))
    (if (null? char-list)
	""
	(list->string 
	 (cons (char-upcase (car char-list))
	       (map char-downcase (cdr char-list)))))))

(define (string-downcase string)
  (list->string (map char-downcase (string->list string))))

;; "Uberschreiben der Operatorpriorit"atsermittlung

(define (operator-priority-plogic expr)
  (cond ((forall-quantification? expr) 9)
	((exists-quantification? expr) 9)
	(else (operator-priority-logic expr))))

(define operator-priority operator-priority-plogic)

;;;
;;; Externe Repr"asentation: \TeX{}Form
;;;

;; "Uberschreiben der Repr"asentation von Aussagenlogik"=Atomen

(define (texform-atom expr)
  (cond ((forall-quantification? expr) 
	 (texform-quantification 
	  "\\forall " 
	  (quantified-variable expr)
	  (texform-aux-open (subject-to-quantification expr) 9)))
	((exists-quantification? expr)
	 (texform-quantification 
	  "\\exists "
	  (quantified-variable expr)
	  (texform-aux-open (subject-to-quantification expr) 9)))
	((predicate-application? expr)
	 (texform-application 
	  (texform-predicate-name (applied-symbol-name expr))
	  (term-list expr)))
	(else (atom->string expr))))
	 
(define (texform-term expr)
  (cond ((variable? expr) 
	 (texform-variable (variable-name expr)))
	((function-application? expr)
	 (if (null? (term-list expr))
	     (texform-constant (applied-symbol-name expr))
	     (texform-application 
	      (texform-function-name (applied-symbol-name expr))
	      (term-list expr))))
	(else (atom->string expr))))
	
(define (texform-predicate-name name)
  (string-append "\\mathit{" 
		 (string-capitalize (atom->string name)) "}"))

(define (texform-function-name name)
  (string-append "\\mathit{" 
		 (string-downcase (atom->string name)) "}"))

(define (texform-variable name)
  (string-append "\\mathit{" 
		 (string-downcase (atom->string name)) "}"))

(define (texform-constant name)
  (string-append "\\mathit{"
		 (string-capitalize (atom->string name)) "}"))

(define (texform-application name-string term-list)
  (cond ((null? term-list) name-string)
	(else (string-append name-string 
			     "(" 
			     (texform-term (car term-list))
			     (texform-term-list "," (cdr term-list))
			     ")"))))

(define (texform-term-list opname oplist)
  (if (null? oplist)
      ""
      (string-append opname
		     (texform-term (car oplist))
		     (texform-term-list opname (cdr oplist)))))
 
(define (texform-quantification quantname quantvar quantexpr-string)
  (string-append quantname 
		 (texform-variable quantvar)
		 "\\, "
		 quantexpr-string))

;;;
;;; Externe Repr"asentation: TextForm
;;;

;; "Uberschreiben der Repr"asentation von Aussagenlogik-Atomen

(define (textform-atom expr)
  (cond ((forall-quantification? expr) 
	 (textform-quantification 
	  "forall " 
	  (quantified-variable expr)
	  (textform-aux-open (subject-to-quantification expr) 9)))
	((exists-quantification? expr)
	 (textform-quantification 
	  "exists "
	  (quantified-variable expr)
	  (textform-aux-open (subject-to-quantification expr) 9)))
	((predicate-application? expr)
	 (textform-application 
	  (textform-predicate-name (applied-symbol-name expr))
	  (term-list expr)))
	(else (atom->string expr))))

(define (textform-term expr)
  (cond ((variable? expr) 
	 (textform-variable (variable-name expr)))
	((function-application? expr)
	 (if (null? (term-list expr))
	     (textform-constant (applied-symbol-name expr))
	     (textform-application 
	      (textform-function-name (applied-symbol-name expr))
	      (term-list expr))))
	(else (atom->string expr))))
	
(define (textform-predicate-name name)
  (string-capitalize (atom->string name)))

(define (textform-function-name name)
  (string-downcase (atom->string name)))

(define (textform-variable name)
  (string-downcase (atom->string name)))

(define (textform-constant name)
  (string-capitalize (atom->string name)))

(define (textform-application name-string term-list)
  (cond ((null? term-list) name-string)
	(else (string-append 
	       name-string "(" 
	       (textform-term (car term-list))
	       (textform-term-list "," (cdr term-list))
	       ")"))))

(define (textform-term-list opname oplist)
  (if (null? oplist)
      ""
      (string-append opname
		     (textform-term (car oplist))
		     (textform-term-list opname (cdr oplist)))))
 
(define (textform-quantification quantname 
				 quantvar quantexpr-string)
  (string-append quantname 
		 (textform-variable quantvar)
		 " "
		 quantexpr-string))

;;;
;;; Substitutionen
;;;

(define make-atomic-substitution cons)
(define atomic-substitution-variable car)
(define atomic-substitution-term cdr)

;; Eine Substitution ist eine Liste atomarer Substitutionen.

(define (make-substitution var term) 
  (list (make-atomic-substitution var term)))

(define (find-substitution var subst)
  ;; Findet die Teilsubstitution, deren erste atomare Substitution
  ;; die angegebene Variable ersetzt.
  (assoc var subst))

(define (compose-substitutions . subst-list)
  ;; Erzeugt die Komposition einer Liste von Substitutionen.
  ;; Die am weitesten links stehende Substitution wird zuletzt auf eine 
  ;; Formel angewandt, wie man es bei der Komposition von Abbildungen 
  ;; gew"ohnt ist.
  ;; Auf die rechten Ersatz-Ausdr"ucke werden also weiter links stehende
  ;; Substitutionen angewandt.
  (if (null? subst-list) 
      '()
      (append 
       (car subst-list)
       (apply 
	compose-substitutions 
	(map (lambda(subst)
	       (map (lambda(atomic-subst) 
		      (make-atomic-substitution
		       (atomic-substitution-variable atomic-subst)
		       (apply-substitution-term 
			(atomic-substitution-term atomic-subst)
			(car subst-list))))
		    subst))
	     (cdr subst-list))))))

(define (apply-substitution formula subst)
  ;; Wendet eine Substitution auf eine Formel an.
  ;; Vorausgesetzt wird eine bereinigte Form, so da"s wir uns
  ;; nicht um gebundene Variablen k"ummern m"ussen.
  (cond ((forall-quantification? formula)
	 (make-forall-quantification 
	  (quantified-variable formula)
	  (apply-substitution (subject-to-quantification formula) 
			      subst)))
	((exists-quantification? formula)
	 (make-exists-quantification 
	  (quantified-variable formula)
	  (apply-substitution (subject-to-quantification formula) 
			      subst)))
	((conjunction? formula)
	 (apply make-conjunction
		(map (lambda(f) (apply-substitution f subst))
		     (subexpr-list formula))))
	((disjunction? formula)
	 (apply make-disjunction
		(map (lambda(f) (apply-substitution f subst))
		     (subexpr-list formula))))
	((negation? formula)
	 (make-negation (apply-substitution (subexpr formula) 
					    subst)))
	((predicate-application? formula)
	 (make-predicate-application 
	  (applied-symbol-name formula)
	  (map (lambda(t) (apply-substitution-term t subst))
	       (term-list formula))))
	(else formula)))

(define (apply-substitution-term term subst)
  ;; Wendet eine Substitution auf einen Term an.
  (cond ((function-application? term)
	 (make-function-application 
	  (applied-symbol-name term)
	  (map (lambda (t) (apply-substitution-term t subst))
	       (term-list term))))
	((variable? term)
	 (let ((s (find-substitution (variable-name term) subst)))
	   (if (pair? s)
	       (atomic-substitution-term s) 
	       term)))
	(else term)))

;;;
;;; Erzeugen von Namen
;;;

(define temp-counter 0)

(define (temporary-symbol prefix)
  (set! temp-counter (+ temp-counter 1))
  (string->symbol 
   (string-append prefix (number->string temp-counter))))

(define (temporary-predicate) 
  (temporary-symbol "p_"))

(define (temporary-variable)
  (temporary-symbol "v_"))

(define (temporary-function)
  (temporary-symbol "f_"))

;;;
;;; Konjunktive Normalform
;;;

(define (canonicalize-atom expr)
  ;; "Uberschreibt \texttt{canonicalize-atom} aus \texttt{logic.scm}.
  ;; Verallgemeinert so die Funktion \texttt{canonicalize}, 
  ;; indem aussagenlogische Atome,
  ;; die pr"adikatenlogische Formeln sind, weiter zerlegt werden.
  (cond ((forall-quantification? expr)
	 (make-forall-quantification 
	  (quantified-variable expr)
	  (canonicalize (subject-to-quantification expr))))
	((exists-quantification? expr)
	 (make-exists-quantification 
	  (quantified-variable expr)
	  (canonicalize (subject-to-quantification expr))))
	(else expr)))
	
(define (rename-bound-variables formula)
  ;; "Uberf"uhrt eine kanonische Formel in eine bereinigte Form
  ;; durch systematisches Umbenennen gebundener Variablen.
  (rename-bound-variables-aux formula '()))

(define (rename-bound-variables-aux formula subst)
  (cond ((negation? formula) 
	 (make-negation 
	  (rename-bound-variables-aux (subexpr formula) subst)))
	((conjunction? formula)
	 (apply make-conjunction 
		(map (lambda(f) 
		       (rename-bound-variables-aux f subst))
		     (subexpr-list formula))))
	((disjunction? formula)
	 (apply make-disjunction
		(map (lambda(f) 
		       (rename-bound-variables-aux f subst))
		     (subexpr-list formula))))
	((forall-quantification? formula)
	 (mangle-quantification make-forall-quantification 
				formula subst))
	((exists-quantification? formula)
	 (mangle-quantification make-exists-quantification 
				formula subst))
	((predicate-application? formula)
	 (make-predicate-application 
	  (applied-symbol-name formula)
	  (map (lambda(term) (apply-substitution-term term subst))
	       (term-list formula))))
	(else formula)))

(define (mangle-quantification make-quantification formula subst)
  (let ((mangled-quantified-variable (temporary-variable)))
    (make-quantification 
     mangled-quantified-variable
     (rename-bound-variables-aux 
      (subject-to-quantification formula)
      (compose-substitutions
       (make-substitution 
	(quantified-variable formula) 
	(make-variable mangled-quantified-variable))
       subst)))))

(define (quantify-free-variables formula)
  ;; Bindet freie Variablen in der kanonischen Formel 
  ;; mit Existenzquantoren.
  (let ((var-list (free-variables formula)))
    (quantify-list formula var-list)))

(define (quantify-list formula var-list)
  (if (null? var-list)
      formula
      (quantify-list 
       (make-exists-quantification (car var-list) formula)
       (cdr var-list))))

(define (free-variables formula)
  ;; Ermittelt eine Liste der freien Variablen in der 
  ;; kanonischen Formel.
  (cond ((negation? formula) (free-variables (subexpr formula)))
	((or (conjunction? formula) 
	     (disjunction? formula))
	 (apply set-append 
		(map free-variables (subexpr-list formula))))
	((predicate-application? formula)
	 (apply set-append 
		(map free-variables-term (term-list formula))))
	((or (exists-quantification? formula)
	     (forall-quantification? formula))
	 (remove-if (lambda(x) 
		      (equal? x (quantified-variable formula)))
		    (free-variables 
		     (subject-to-quantification formula))))
	(else '())))

(define (free-variables-term term)
  (cond ((function-application? term)
	 (apply set-append 
		(map free-variables-term (term-list term))))
	((variable? term)
	 (list (variable-name term)))
	(else '())))

(define (replace-predicate formula predicate replacement)
  ;; Ersetzt in der Formel die Pr"adikat"=Anwendung durch die
  ;; Ersatzformel; kennt nur Quantifizierungen und Negation.
  (cond ((forall-quantification? formula)
	 (make-forall-quantification 
	  (quantified-variable formula)
	  (replace-predicate (subject-to-quantification formula) 
			     predicate replacement)))
	((exists-quantification? formula) 
	 (make-exists-quantification 
	  (quantified-variable formula)
	  (replace-predicate (subject-to-quantification formula) 
			     predicate replacement)))
	((negation? formula)
	 (make-negation (replace-predicate (subexpr formula)
					   predicate replacement)))
	((predicate-application? formula)
	 (if (equal? (applied-symbol-name formula) predicate)
	     replacement
	     formula))
	(else formula)))

(define (prenexify formula)
  ;; "Uberf"uhrt eine kanonische, bereinigte, vollst"andig gebundene Formel
  ;; in die Pr"anexform.
  (let* ((temp-predicate (temporary-predicate))
	 (collected 
	  (collect-quantifications 
	   formula 
	   (make-predicate-application temp-predicate '()))))
    (replace-predicate (cdr collected) 
		       temp-predicate 
		       (car collected))))

(define (collect-quantifications formula temp-predicate)
  ;; Liefert ein Paar: Dequantifizierte Formel und Quantoren-Vorlage
  ;; (das angegebene Pr"adikat, quantifiziert mit allen Quantoren).
  (cond ((negation? formula) 
	 (let ((collected (collect-quantifications (subexpr formula)
						   temp-predicate)))
	   (cons (make-negation (car collected))
		 (morganize-negation (cdr collected)))))
	((disjunction? formula)
	 (collect-quantifications-multiop 
	  make-disjunction (subexpr-list formula) temp-predicate))
	((conjunction? formula)
	 (collect-quantifications-multiop 
	  make-conjunction (subexpr-list formula) temp-predicate))
	((forall-quantification? formula)
	 (swap-quantification formula 
			      make-forall-quantification 
			      temp-predicate))
	((exists-quantification? formula)
	 (swap-quantification formula
			      make-exists-quantification 
			      temp-predicate))
	(else (cons formula temp-predicate))))

(define (collect-quantifications-multiop make-multiop 
					 expr-list temp-predicate)
  (collect-quantifications-list make-multiop expr-list '() 
				temp-predicate temp-predicate))

(define (collect-quantifications-list make-multiop 
				      expr-list changed-list 
				      quant temp-predicate)
  (if (null? expr-list) 
      (cons (apply make-multiop changed-list)
	    quant)
      (let ((collected-car 
	     (collect-quantifications (car expr-list) 
				      temp-predicate)))
	(collect-quantifications-list 
	 make-multiop
	 (cdr expr-list) 
	 (cons (car collected-car) changed-list)
	 (replace-predicate quant 
			    (applied-symbol-name temp-predicate)
			    (cdr collected-car))
	 temp-predicate))))

(define (swap-quantification formula 
			     make-quantification temp-predicate)
  ;; W"alzt einen Quantor der Formel auf die Quantoren-Vorlage
  ;; der gebundenen Formel um.
  (let ((collected (collect-quantifications 
		    (subject-to-quantification formula)
		    temp-predicate)))
    (cons (car collected)
	  (make-quantification (quantified-variable formula)
			       (cdr collected)))))

(define (morganize-negation formula)
  ;; Morganisiert alle f"uhrenden Quantoren.
  (cond ((forall-quantification? formula) 
	 (make-exists-quantification 
	  (quantified-variable formula)
	  (morganize-negation (subject-to-quantification formula))))
	((exists-quantification? formula)
	 (make-forall-quantification
	  (quantified-variable formula)
	  (morganize-negation (subject-to-quantification formula))))
	(else formula)))

(define (skolemify formula)
  ;; "Uberf"uhrt eine Formel in Pr"anexform in die Skolemform
  (skolemify-aux formula '() '()))

(define (skolemify-aux formula variable-list subst)
  ;; Sammelt die all-quantifizierten Variablen in einer Variablenliste
  ;; und ersetzt alle Vorkommen von Existenz-quantifizierten Variablen 
  ;; durch eine Anwendung eines neu erfundenen Funktionssymbols auf
  ;; die gesammelten Variablen.
  (cond ((forall-quantification? formula)
	 (make-forall-quantification
	  (quantified-variable formula)
	  (skolemify-aux 
	   (subject-to-quantification formula)
	   (cons (make-variable 
		  (quantified-variable formula))
		 variable-list)
	   subst)))
	((exists-quantification? formula)
	 (skolemify-aux 
	  (subject-to-quantification formula)
	  variable-list
	  (compose-substitutions 
	   (make-substitution 
	    (quantified-variable formula)
	    (make-function-application (temporary-function)
				       variable-list)))))
	(else (apply-substitution formula subst))))

(define (normalize formula)
  ;; Bringt eine Formel in die KNF. "Uberschreibt die gleichnamige
  ;; Funktion aus \texttt{logic.scm}.
  (remove-duplicates 
   (flatten 
    (normalize-canonical
     (dequantify
      (pre-normalize formula))))))

(define (pre-normalize formula)
  (skolemify 
   (prenexify 
    (quantify-free-variables 
     (rename-bound-variables 
      (canonicalize formula))))))

(define (dequantify formula)
  ;; Entfernt f"uhrende Quantoren.
  (cond ((or (forall-quantification? formula)
	     (exists-quantification? formula)) 
	 (dequantify (subject-to-quantification formula)))
	(else formula)))

;;;
;;; Unifikation von Termen und Pr"adikat"=Anwendungen
;;;

(define (occurrence? variable term)
  ;; Implementation des `occur check'.
  (cond ((function-application? term)
	 (apply lambda-or (map (lambda(t) (occurrence? variable t))
			       (term-list term))))
	((variable? term)
	 (equal? (variable-name term) variable))))

(define (difference predicate-app-list)
  ;; Ermittelt eine zur Eingabeliste parallele Liste,
  ;; welche die Differenzterme enth"alt. Das Ergebnis ist
  ;; \texttt{\#t}, wenn die Pr"adikat"=Anwendungen identisch sind;
  ;; \texttt{()}, wenn wegen unterschiedlicher Pr"adikate nichts
  ;; machbar ist. Ansonsten wird die Liste der Unterscheidungsterme 
  ;; geliefert.
  (let ((car-arity (arity (car predicate-app-list)))
	(car-name (applied-symbol-name (car predicate-app-list))))
    (if (apply 
	 lambda-and 
	 (map (lambda (pred-app) 
		(and 
		 (equal? (applied-symbol-name pred-app) car-name)
		 (= (arity pred-app) car-arity)))
	      (cdr predicate-app-list)))
	(difference-list (map term-list predicate-app-list))
	'())))

(define (difference-list term-list-list)
  ;; Liefert die Liste der Unterscheidungsterme.
  ;; Vorausgesetzt wird, da"s eine Liste gleichlanger
  ;; Listen vorliegt.
  (if (null? (car term-list-list))
      #t
      (let ((mapcar-difference 
	     (difference-term (map car term-list-list))))
	(if (eq? mapcar-difference #t)
	    (difference-list (map cdr term-list-list))
	    mapcar-difference))))

(define (difference-term t-list)
  ;; Liefert die Liste der Unterscheidungsterme
  ;; oder \texttt{\#t}, wenn die Terme identisch sind.
  (cond ((function-application? (car t-list))
	 ;; "Uberpr"ufe, ob alle Elemente Funktionsanwendungen sind,
	 ;; deren Funktionssymbole zudem "ubereinstimmen;
	 ;; wenn ja, gehe deren Argumente von links nach rechts durch.
	 ;; Ansonsten liefere die ganze Liste als Differenzliste.
	 (let ((car-arity (arity (car t-list)))
	       (car-name (applied-symbol-name (car t-list))))
	   (if (apply lambda-and 
		      (map (lambda (t) 
			     (and (function-application? t)
				  (equal? (applied-symbol-name t) 
					  car-name)
				  (= (arity t) car-arity)))
			   (cdr t-list)))
	       (difference-list (map term-list t-list))
	       t-list)))
	((variable? (car t-list))
	 ;; "Uberpr"ufe, ob alle Variablen "ubereinstimmen;
	 ;; wenn ja, liefere \texttt{\#t}. 
	 ;; Ansonsten liefere die ganze Liste als Differenzliste.
	 (let ((car-name (variable-name (car t-list))))
	   (if (apply 
		lambda-and 
		(map (lambda (t)
		       (and (variable? t)
			    (equal? (variable-name t) car-name)))
		     (cdr t-list)))
	       #t
	       t-list)))
	(else t-list)))
		    
(define (mgu predicate-app-list)
  ;; Ermittelt den `mgu' (allgemeinsten Unifikator) der 
  ;; Pr"adikat"=Anwendungen
  ;; mittels des Verfahrens von Robinson.
  ;; Das Ergebnis ist \texttt{\#t}, wenn die 
  ;; Pr"adikat"=Anwendungen nicht 
  ;; unifizierbar sind; ansonsten die gefundene Substitution.
  (mgu-aux predicate-app-list '()))

(define (mgu-aux predicate-app-list subst)
  (let ((diff-list (difference predicate-app-list)))
    (cond 
     ((eq? diff-list #t) subst)
     ((null? diff-list) #t)
     (else 
      (let ((list-var (first-that variable? diff-list)))
	(cond 
	 ((null? list-var) #t)
	 ;; Mehrfaches Auftreten der Variable f"ur sich
	 ;; ist zul"assig; jedoch nicht Auftreten der
	 ;; Variable innerhalb der Terme (`occur check'). 
	 ((apply lambda-or 
		 (map (lambda (t) 
			(occurrence? 
			 (variable-name (car list-var)) 
			 t))
		      (remove-if variable? diff-list)))
	  #t)
	 (else 
	  ;; W"ahle einen Term, der nicht die Variable selbst
	  ;; ist, erstelle eine Substitution und wende sie an.
	  (let* ((list-notvar 
		  (first-that 
		   (lambda (t)
		     (or (not (variable? t))
			 (not 
			  (eq? (variable-name t)
			       (variable-name (car list-var))))))
		   diff-list))
		 (new-subst 
		  (make-substitution 
		   (variable-name (car list-var))
		   (car list-notvar))))
	    (mgu-aux 
	     (make-set 
	      (map (lambda (f)
		     (apply-substitution f new-subst))
		   predicate-app-list))
	     (compose-substitutions new-subst subst))))))))))

(define (mgu-term term-list)
  ;; Das ist die in der Ausgabenstellung beschriebene mgu-Funktion,
  ;; welche auf einer Liste von \emph{Termen} operiert. Sie wird sonst
  ;; nicht gebraucht.
  (let ((temp-pred (temporary-predicate)))
    (mgu (map (lambda (t) 
		(make-predicate-application temp-pred (list t)))
	      term-list))))

;;; 
;;; Pr"adikatenlogische Resolution
;;;

;; Wir verwenden die gleiche Resolutionsstrategie wie bei der
;; Aussagenlogik: Es wird eine Breitensuche durchgef"uhrt.
;; Man beachte, da"s die Funktion \texttt{resolvent-diagonal?} in der
;; Pr"adikatenlogik nur ein Semi"=Entscheidungsverfahren ist.
;; -- Wir m"ussen nur eine einzige Funktion "uberschreiben:

(define (resolve-all-list list1-to-come list1-done list2 res-list)
  ;; "Uberschreibt die Funktion aus \texttt{logic.scm}.
  (cond ((null? list1-to-come) res-list)
	(else (resolve-all-list 
	       (cdr list1-to-come)
	       (cons (car list1-to-come) list1-done)
	       list2
	       (resolve-all-list-aux
		(car list1-to-come) 
		(append (cdr list1-to-come) list1-done)
		list2 
		'()
		res-list)))))

(define (resolve-all-list-aux list1-cur list1-rest
			      list2-to-come list2-done res-list)
  (cond ((null? list2-to-come) res-list)
	((contrary? list1-cur (car list2-to-come))
	 (let ((car-mgu 
		(mgu (list (literal-atom list1-cur)
			   (literal-atom (car list2-to-come))))))
	   ;; Wenn nicht unifizierbar, in der zweiten Liste
	   ;; weitersuchen. Ansonsten Resolvente speichern
	   ;; und weitersuchen.
	   (cond ((eq? car-mgu #t) 
		  (resolve-all-list-aux 
		   list1-cur list1-rest
		   (cdr list2-to-come) 
		   (cons (car list2-to-come) list2-done)
		   res-list))
		 (else 
		  (resolve-all-list-aux
		   list1-cur list1-rest
		   (cdr list2-to-come) 
		   (cons (car list2-to-come) list2-done)
		   (cons (map (lambda(literal)
				(apply-substitution literal 
						    car-mgu))
			      (join-literal-lists
			       (cons list1-cur list1-rest)
			       (append list2-to-come list2-done)
			       list1-cur
			       (car list2-to-come)))
			 res-list))))))
	(else (resolve-all-list-aux 
	       list1-cur list1-rest
	       (cdr list2-to-come) 
	       (cons (car list2-to-come) list2-done)
	       res-list))))

;;;
;;; Antwortsystem
;;;

(define (make-answer var-name)
  (make-predicate-application 
   'answer 
   (if (pair? var-name)
       (map make-variable var-name)
       (list (make-variable var-name)))))

(define (answer? formula)
  (and (predicate-application? formula)
       (equal? (applied-symbol-name formula) 'answer)))

(define (answer-clause? formula)
  ;; Ermittelt, ob die Klausel nur aus Antwortliteralen
  ;; besteht. Darin enthalten ist der Fall der leeren Klausel.
  (apply lambda-and (map answer? (subexpr-list formula))))

;; Wir redefinieren die Entscheidung, ob eine 
;; Antwort gefunden wurde; falls ja, geben wir
;; auch die Antwort zur"uck.

(define (found-answer? res-list)
  ;; Ermittelt, ob eine Antwort-Klausel (eine Klausel, die nur
  ;; Antwortterme enth"alt) in der Liste enthalten ist.
  (cond ((null? res-list) #f)
	((answer-clause? (car res-list)) (car res-list))
	(else (found-answer? (cdr res-list)))))

(define (forall-quantify var-name formula)
  (cond ((null? var-name) formula)
	((pair? var-name) 
	 (forall-quantify 
	  (cdr var-name) 
	  (make-forall-quantification (car var-name) formula)))
	(else (make-forall-quantification var-name formula))))

(define (answer axioms question variable)
  ;; Ermittelt eine Antwort auf die Frage.
  ;; Es kann eine einzelne Variable oder eine Liste
  ;; von Variablen, die in der Frage frei sind,
  ;; angegeben werden.
  (contradiction? (normalize
		   (make-conjunction 
		    axioms 
		    (make-forall-quantification 
		     variable
		     (make-disjunction
		      (make-negation question)
		      (make-answer variable)))))))
