;;; bpp.el --- Highlight or hide text according to Pascal conditionals.
;; Derived from cpp.el of Emacs 19.34; original message follows.
;; MODIFIED by <mkoeppe@cs.uni-magdeburg.de> 

;; Copyright (C) 1994, 1995 Free Software Foundation

;; Author: Per Abrahamsen <abraham@iesd.auc.dk>
;; Keywords: c, faces, tools

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Parse a text for C preprocessor conditionals, and highlight or hide
;; the text inside the conditionals as you wish.

;; This package is inspired by Jim Coplien's delta editor for SCCS.

;;; Todo:

;; Should parse "#if" and "#elif" expressions and merge the faces
;; somehow. 

;; Somehow it is sometimes possible to make changes near a read only
;; area which you can't undo.  Their are other strange effects in that
;; area.

;; The Edit buffer should -- optionally -- appear in its own frame.

;; Conditionals seem to be rear-sticky.  They shouldn't be.

;; Restore window configurations when exiting CPP Edit buffer.

;;; Code:

;;; Customization:

(defvar bpp-config-file (convert-standard-filename "~/_bpp.el")
  "*File name to save bpp configuration.")

(defvar bpp-known-face 'invisible
  "*Face used for known bpp symbols.")

(defvar bpp-unknown-face 'highlight
  "*Face used for unknown bpp symbols.")

(defvar bpp-face-type 'light 
  "*Indicate what background face type you prefer.
Can be either light or dark for color screens, mono for monochrome
screens, and none if you don't use a window system.")

(defvar bpp-known-writable t
  "*Non-nil means you are allowed to modify the known conditionals.")

(defvar bpp-unknown-writable t
  "*Non-nil means you are allowed to modify the unknown conditionals.")

(defvar bpp-edit-list nil
  "Alist of bpp macros and information about how they should be displayed.
Each entry is a list with the following elements:
0. The name of the macro (a string).
1. Face used for text that is `ifdef' the macro.
2. Face used for text that is `ifndef' the macro.
3. `t', `nil', or `both' depending on what text may be edited.")

(defvar bpp-overlay-list nil)
;; List of bpp overlays active in the current buffer.
(make-variable-buffer-local 'bpp-overlay-list)

(defvar bpp-callback-data)
(defvar bpp-state-stack)

(defconst bpp-face-type-list
  '(("light color background" . light)
    ("dark color background" . dark)
    ("monochrome" . mono)
    ("tty" . none))
  "Alist of strings and names of the defined face collections.")

(defconst bpp-writable-list
  ;; Names used for the writable property.
  '(("writable" . t)
    ("read-only" . nil)))

(defvar bpp-button-event nil)
;; This will be t in the callback for `bpp-make-button'.

(defvar bpp-edit-buffer nil)
;; Real buffer whose bpp display information we are editing.
(make-variable-buffer-local 'bpp-edit-buffer)

(defconst bpp-branch-list
  ;; Alist of branches.
  '(("false" . nil)
    ("true" . t)
    ("both" . both)))

(defvar bpp-face-default-list nil
  "List of faces you can choose from for bpp conditionals.")

(defvar bpp-face-light-name-list
  '("light gray" "light blue" "light cyan" "light yellow" "light pink"
    "pale green" "beige" "orange" "magenta" "violet" "medium purple"
    "turquoise")
  "Background colours useful with dark foreground colors.")

(defvar bpp-face-dark-name-list
  '("dim gray" "blue" "cyan" "yellow" "red"
    "dark green" "brown" "dark orange" "dark khaki" "dark violet" "purple"
    "dark turquoise")
  "Background colours useful with light foreground colors.")

(defvar bpp-face-light-list nil
  "Alist of names and faces to be used for light backgrounds.")

(defvar bpp-face-dark-list nil
  "Alist of names and faces to be used for dark backgrounds.")

(defvar bpp-face-mono-list
  '(("bold" . 'bold)
    ("bold-italic" . 'bold-italic)
    ("italic" . 'italic)
    ("underline" . 'underline))
  "Alist of names and faces to be used for monochrome screens.")

(defvar bpp-face-none-list
   '(("default" . default)
     ("invisible" . invisible))
   "Alist of names and faces available even if you don't use a window system.")

(defvar bpp-face-all-list
  (append bpp-face-light-list
	  bpp-face-dark-list
	  bpp-face-mono-list
	  bpp-face-none-list)
  "All faces used for highlighting text inside bpp conditionals.")

;;; Parse Buffer:

(defvar bpp-parse-symbols nil
  "List of bpp macros used in the local buffer.")
(make-variable-buffer-local 'bpp-parse-symbols)

(defconst bpp-parse-regexp
  ;; Regexp matching all tokens needed to find conditionals.
  (concat
   "\\({\\$\\([iI][fF][dD][eE][fF]\\|[iI][fF][nN][dD][eE][fF]\\|[eE][lL][sS][eE]\\|[eE][nN][dD][iI][fF]\\)[ \\t]*\\(\\w*\\)[^}]*}\\)\\|"   
   "'\\|{"))

;;;###autoload
(defun bpp-highlight-buffer (arg)
  "Highlight C code according to preprocessor conditionals.
This command pops up a buffer which you should edit to specify
what kind of highlighting to use, and the criteria for highlighting.
A prefix arg suppresses display of that buffer."
  (interactive "P")
  (setq bpp-parse-symbols nil)
  (bpp-parse-reset)
  (if (null bpp-edit-list)
      (bpp-edit-load))
  (let (bpp-state-stack)
    (save-excursion
      (goto-char (point-min))
      (bpp-progress-message "Parsing...")
      (while (re-search-forward bpp-parse-regexp nil t)
	(bpp-progress-message "Parsing...%d%%"
			  (/ (* 100 (- (point) (point-min))) (buffer-size)))
	(let ((match (buffer-substring (match-beginning 0) (match-end 0))))
	  (cond ((or (string-equal match "'")
;		     (string-equal match "\"")
		     )
		 (goto-char (match-beginning 0))
		 (condition-case nil
		     (forward-sexp)
		   (error (bpp-parse-error
			   "Unterminated string"))))
		((string-equal match "{")
		 (or (search-forward "}" nil t)
		     (error "Unterminated comment")))
;		((string-equal match "//")
;		 (skip-chars-forward "^\n\r"))
		(t
		 ;(end-of-line 1)
		 (goto-char (match-end 1))
		 (let ((from (match-beginning 1))
		       (to (point))
		       (type (buffer-substring (match-beginning 2)
					       (match-end 2)))
		       (expr (upcase (buffer-substring (match-beginning 3) 
					       (match-end 3)))))
		   (cond ((string-equal (upcase type) "IFDEF")
			  (bpp-parse-open t expr from to))
			 ((string-equal (upcase type) "IFNDEF")
			  (bpp-parse-open nil expr from to))
			 ((string-equal (upcase type) "IF")
			  (bpp-parse-open t expr from to))
;			 ((string-equal type "elif")
;			  (let (bpp-known-face bpp-unknown-face)
;			    (bpp-parse-close from to))
;			  (bpp-parse-open t expr from to))
			 ((string-equal (upcase type) "ELSE")
			  (or bpp-state-stack
			      (bpp-parse-error "Top level $else"))
			  (let ((entry (list (not (nth 0 (car bpp-state-stack)))
					     (nth 1 (car bpp-state-stack))
					     from to)))
			    (bpp-parse-close from to)
			    (setq bpp-state-stack (cons entry bpp-state-stack))))
			 ((string-equal (upcase type) "ENDIF")
			  (bpp-parse-close from to))
			 (t
			  (bpp-parse-error "Parser error"))))))))
      (message "Parsing...done"))
    (if bpp-state-stack
      (save-excursion
	(goto-char (nth 3 (car bpp-state-stack)))
	(bpp-parse-error "Unclosed conditional"))))
  (or arg
      (null bpp-parse-symbols)
      (bpp-parse-edit)))

(defun bpp-parse-open (branch expr begin end)
  "Push information about conditional-beginning onto `bpp-state-stack'."
  ;; Discard comments within this line.
  (while (string-match "\\b[ \t]*{.*}[ \t]*\\b" expr)
    (setq expr (concat (substring expr 0 (match-beginning 0))
		       (substring expr (match-end 0)))))
  ;; If a comment starts on this line and continues past, discard it.
  (if (string-match "\\b[ \t]*{" expr)
      (setq expr (substring expr 0 (match-beginning 0))))
  ;; Delete any C++ comment from the line.
;  (if (string-match "\\b[ \t]*\\(//.*\\)?$" expr)
;      (setq expr (substring expr 0 (match-beginning 0))))
  (while (string-match "[ \t]+" expr)
      (setq expr (concat (substring expr 0 (match-beginning 0))
			 (substring expr (match-end 0)))))
  (setq bpp-state-stack (cons (list branch expr begin end) bpp-state-stack))
  (or (member expr bpp-parse-symbols)
      (setq bpp-parse-symbols
	    (cons expr bpp-parse-symbols)))
  (if (assoc expr bpp-edit-list)
      (bpp-make-known-overlay begin end)
    (bpp-make-unknown-overlay begin end)))

(defun bpp-parse-close (from to)
  ;; Pop top of bpp-state-stack and create overlay.
  (let ((entry (assoc (nth 1 (car bpp-state-stack)) bpp-edit-list))
	(branch (nth 0 (car bpp-state-stack)))
	(begin (nth 2 (car bpp-state-stack)))
	(end (nth 3 (car bpp-state-stack))))
    (setq bpp-state-stack (cdr bpp-state-stack))
    (if entry
	(let ((face (nth (if branch 1 2) entry))
	      (read-only (eq (not branch) (nth 3 entry)))
	      (priority (length bpp-state-stack))
	      (overlay (make-overlay end from)))
	  (bpp-make-known-overlay from to)
	  (setq bpp-overlay-list (cons overlay bpp-overlay-list))
	  (if priority (overlay-put overlay 'priority priority))
	  (cond ((eq face 'invisible)
		 (bpp-make-overlay-hidden overlay))
		((eq face 'default))
		(t
		 (overlay-put overlay 'face face)))
	  (if read-only
	      (bpp-make-overlay-read-only overlay)
	    (bpp-make-overlay-sticky overlay)))
      (bpp-make-unknown-overlay from to))))

(defun bpp-parse-error (error)
  ;; Error message issued by the bpp parser.
  (error "%s at line %d" error (count-lines (point-min) (point))))

(defun bpp-parse-reset ()
  "Reset display of bpp conditionals to normal."
  (interactive)
  (while bpp-overlay-list
    (delete-overlay (car bpp-overlay-list))
    (setq bpp-overlay-list (cdr bpp-overlay-list))))

;;;###autoload
(defun bpp-parse-edit ()
  "Edit display information for bpp conditionals."
  (interactive)
  (or bpp-parse-symbols
      (bpp-highlight-buffer t))
  (let ((buffer (current-buffer)))
    (pop-to-buffer "*BPP Edit*")
    (bpp-edit-mode)
    (setq bpp-edit-buffer buffer)
    (bpp-edit-reset)))

;;; Overlays:

(defun bpp-make-known-overlay (start end)
  ;; Create an overlay for a known bpp command from START to END.
  (let ((overlay (make-overlay start end)))
    (if (eq bpp-known-face 'invisible)
	(bpp-make-overlay-hidden overlay)
      (or (eq bpp-known-face 'default)
	  (overlay-put overlay 'face bpp-known-face))
      (if bpp-known-writable
	  ()
	(overlay-put overlay 'modification-hooks '(bpp-signal-read-only))
	(overlay-put overlay 'insert-in-front-hooks '(bpp-signal-read-only))))
    (setq bpp-overlay-list (cons overlay bpp-overlay-list))))

(defun bpp-make-unknown-overlay (start end)
  ;; Create an overlay for an unknown bpp command from START to END.
  (let ((overlay (make-overlay start end)))
    (cond ((eq bpp-unknown-face 'invisible)
	   (bpp-make-overlay-hidden overlay))
	  ((eq bpp-unknown-face 'default))
	  (t 
	   (overlay-put overlay 'face bpp-unknown-face)))
    (if bpp-unknown-writable
	()
      (overlay-put overlay 'modification-hooks '(bpp-signal-read-only))
      (overlay-put overlay 'insert-in-front-hooks '(bpp-signal-read-only)))
    (setq bpp-overlay-list (cons overlay bpp-overlay-list))))

(defun bpp-make-overlay-hidden (overlay)
  ;; Make overlay hidden and intangible.
  (overlay-put overlay 'invisible t)
  (overlay-put overlay 'intangible t)
  ;; Unfortunately `intangible' is not implemented for overlays yet,
  ;; so we make is read-only instead.
  (overlay-put overlay 'modification-hooks '(bpp-signal-read-only))
  (overlay-put overlay 'insert-in-front-hooks '(bpp-signal-read-only)))

(defun bpp-make-overlay-read-only (overlay)
  ;; Make overlay read only.
  (overlay-put overlay 'modification-hooks '(bpp-signal-read-only))
  (overlay-put overlay 'insert-in-front-hooks '(bpp-signal-read-only))
  (overlay-put overlay 'insert-behind-hooks '(bpp-signal-read-only)))

(defun bpp-make-overlay-sticky (overlay)
  ;; Make OVERLAY grow when you insert text at either end.
  (overlay-put overlay 'insert-in-front-hooks '(bpp-grow-overlay))
  (overlay-put overlay 'insert-behind-hooks '(bpp-grow-overlay)))

(defun bpp-signal-read-only (overlay after start end &optional len)
  ;; Only allow deleting the whole overlay.
  ;; Trying to change a read-only overlay.
  (if (and (not after)
	   (or (< (overlay-start overlay) start)
	       (> (overlay-end overlay) end)))
      (error "This text is read only")))

(defun bpp-grow-overlay (overlay after start end &optional len)
  ;; Make OVERLAY grow to contain range START to END.
  (if after
      (move-overlay overlay
		    (min start (overlay-start overlay))
		    (max end (overlay-end overlay)))))

;;; Edit Buffer:

(defvar bpp-edit-map nil)
;; Keymap for `bpp-edit-mode'.

(if bpp-edit-map
    ()
  (setq bpp-edit-map (make-keymap))
  (suppress-keymap bpp-edit-map)
  (define-key bpp-edit-map [ down-mouse-2 ] 'bpp-push-button)
  (define-key bpp-edit-map [ mouse-2 ] 'ignore)
  (define-key bpp-edit-map " " 'scroll-up)
  (define-key bpp-edit-map "\C-?" 'scroll-down)
  (define-key bpp-edit-map [ delete ] 'scroll-down)
  (define-key bpp-edit-map "\C-c\C-c" 'bpp-edit-apply)
  (define-key bpp-edit-map "a" 'bpp-edit-apply)
  (define-key bpp-edit-map "A" 'bpp-edit-apply)
  (define-key bpp-edit-map "r" 'bpp-edit-reset)
  (define-key bpp-edit-map "R" 'bpp-edit-reset)
  (define-key bpp-edit-map "s" 'bpp-edit-save)
  (define-key bpp-edit-map "S" 'bpp-edit-save)
  (define-key bpp-edit-map "l" 'bpp-edit-load)
  (define-key bpp-edit-map "L" 'bpp-edit-load)
  (define-key bpp-edit-map "h" 'bpp-edit-home)
  (define-key bpp-edit-map "H" 'bpp-edit-home)
  (define-key bpp-edit-map "b" 'bpp-edit-background)
  (define-key bpp-edit-map "B" 'bpp-edit-background)
  (define-key bpp-edit-map "k" 'bpp-edit-known)
  (define-key bpp-edit-map "K" 'bpp-edit-known)
  (define-key bpp-edit-map "u" 'bpp-edit-unknown)
  (define-key bpp-edit-map "u" 'bpp-edit-unknown)
  (define-key bpp-edit-map "t" 'bpp-edit-true)
  (define-key bpp-edit-map "T" 'bpp-edit-true)
  (define-key bpp-edit-map "f" 'bpp-edit-false)
  (define-key bpp-edit-map "F" 'bpp-edit-false)
  (define-key bpp-edit-map "w" 'bpp-edit-write)
  (define-key bpp-edit-map "W" 'bpp-edit-write)
  (define-key bpp-edit-map "X" 'bpp-edit-toggle-known)
  (define-key bpp-edit-map "x" 'bpp-edit-toggle-known)
  (define-key bpp-edit-map "Y" 'bpp-edit-toggle-unknown)
  (define-key bpp-edit-map "y" 'bpp-edit-toggle-unknown)
  (define-key bpp-edit-map "q" 'bury-buffer)
  (define-key bpp-edit-map "Q" 'bury-buffer))

(defvar bpp-edit-symbols nil)
;; Symbols defined in the edit buffer.
(make-variable-buffer-local 'bpp-edit-symbols)

(defun bpp-edit-mode ()
  "Major mode for editing the criteria for highlighting bpp conditionals.
Click on objects to change them.  
You can also use the keyboard accelerators indicated like this: [K]ey."
  (kill-all-local-variables)
  (buffer-disable-undo)
  (auto-save-mode -1)
  (setq buffer-read-only t)
  (setq major-mode 'bpp-edit-mode)
  (setq mode-name "BPP Edit")
  (use-local-map bpp-edit-map))

(defun bpp-edit-apply ()
  "Apply edited display information to original buffer."
  (interactive)
  (bpp-edit-home)
  (bpp-highlight-buffer t))

(defun bpp-edit-reset ()
  "Reset display information from original buffer."
  (interactive)
  (let ((buffer (current-buffer))
	(buffer-read-only nil)
	(start (window-start))
	(pos (point))
	symbols)
    (set-buffer bpp-edit-buffer)
    (setq symbols bpp-parse-symbols)
    (set-buffer buffer)
    (setq bpp-edit-symbols symbols)
    (erase-buffer)
    (insert "BPP Display Information for `")
    (bpp-make-button (buffer-name bpp-edit-buffer) 'bpp-edit-home)
    (insert "\n\nClick mouse-2 on item you want to change or use\n"
	    "or switch to this buffer and type the keyboard equivalents.\n"
	    "Keyboard equivalents are indicated with brackets like [T]his.\n\n")
    (bpp-make-button "[H]ome (display the C file)" 'bpp-edit-home)
    (insert "  ")
    (bpp-make-button "[A]pply new settings" 'bpp-edit-apply)
    (insert "\n")
    (bpp-make-button "[S]ave settings" 'bpp-edit-save)
    (insert "  ")
    (bpp-make-button "[L]oad settings" 'bpp-edit-load)
    (insert "\n\n")

    (insert "[B]ackground: ")
    (bpp-make-button (car (rassq bpp-face-type bpp-face-type-list))
		     'bpp-edit-background)
    (insert "\n[K]nown conditionals: ")
    (bpp-make-button (bpp-face-name bpp-known-face)
		     'bpp-edit-known nil t)
    (insert " [X] ")
    (bpp-make-button (car (rassq bpp-known-writable bpp-writable-list))
		     'bpp-edit-toggle-known)
    (insert "\n[U]nknown conditionals: ")
    (bpp-make-button (bpp-face-name bpp-unknown-face)
		     'bpp-edit-unknown nil t)
    (insert " [Y] ")
    (bpp-make-button (car (rassq bpp-unknown-writable bpp-writable-list))
		     'bpp-edit-toggle-unknown)
    (insert (format "\n\n\n%39s: %14s %14s %7s\n\n" "Expression"
		    "[T]rue Face" "[F]alse Face" "[W]rite"))
    (while symbols
      (let*  ((symbol (car symbols))
	      (entry (assoc symbol bpp-edit-list))
	      (true (nth 1 entry))
	      (false (nth 2 entry))
	      (write (if entry (nth 3 entry) 'both)))
	(setq symbols (cdr symbols))

	(if (and entry			; Make default entries unknown.
		 (or (null true) (eq true 'default))
		 (or (null false) (eq false 'default))
		 (eq write 'both))
	    (setq bpp-edit-list (delq entry bpp-edit-list)
		  entry nil))
	
	(if (> (length symbol) 39)
	    (insert (substring symbol 0 39) ": ")
	  (insert (format "%39s: " symbol)))

	(bpp-make-button (bpp-face-name true)
			 'bpp-edit-true symbol t 14)
	(insert " ")
	(bpp-make-button (bpp-face-name false)
			 'bpp-edit-false symbol t 14)
	(insert " ")
	(bpp-make-button (car (rassq write bpp-branch-list))
			 'bpp-edit-write symbol nil 6)
	(insert "\n")))
    (insert "\n\n")
    (set-window-start nil start)
    (goto-char pos)))

(defun bpp-edit-load ()
  "Load bpp configuration."
  (interactive)
  (cond ((file-readable-p bpp-config-file)
	 (load-file bpp-config-file))
	((file-readable-p (concat "~/" bpp-config-file))
	 (load-file bpp-config-file)))
  (if (eq major-mode 'bpp-edit-mode)
      (bpp-edit-reset)))

(defun bpp-edit-save ()
  "Load bpp configuration."
  (interactive)
  (require 'pp)
  (save-excursion
    (set-buffer bpp-edit-buffer)
    (let ((buffer (find-file-noselect bpp-config-file)))
      (set-buffer buffer)
      (erase-buffer)
      (pp (list 'setq 'bpp-known-face
		(list 'quote bpp-known-face)) buffer)
      (pp (list 'setq 'bpp-unknown-face
		(list 'quote bpp-unknown-face)) buffer)
      (pp (list 'setq 'bpp-face-type
		(list 'quote bpp-face-type)) buffer)
      (pp (list 'setq 'bpp-known-writable
		(list 'quote bpp-known-writable)) buffer)
      (pp (list 'setq 'bpp-unknown-writable
		(list 'quote bpp-unknown-writable)) buffer)
      (pp (list 'setq 'bpp-edit-list
		(list 'quote bpp-edit-list)) buffer)
      (write-file bpp-config-file))))

(defun bpp-edit-home ()
  "Switch back to original buffer."
  (interactive)
  (if bpp-button-event
      (read-event))
  (pop-to-buffer bpp-edit-buffer))

(defun bpp-edit-background ()
  "Change default face collection."
  (interactive)
  (call-interactively 'bpp-choose-default-face)
  (bpp-edit-reset))

(defun bpp-edit-known ()
  "Select default for known conditionals."
  (interactive)
  (setq bpp-known-face (bpp-choose-face "Known face" bpp-known-face))
  (bpp-edit-reset))

(defun bpp-edit-unknown ()
  "Select default for unknown conditionals."
  (interactive)
  (setq bpp-unknown-face (bpp-choose-face "Unknown face" bpp-unknown-face))
  (bpp-edit-reset))

(defun bpp-edit-toggle-known (arg)
  "Toggle writable status for known conditionals.
With optional argument ARG, make them writable iff ARG is positive."
  (interactive "@P")
  (if (or (and (null arg) bpp-known-writable)
	  (<= (prefix-numeric-value arg) 0))
      (setq bpp-known-writable nil)
    (setq bpp-known-writable t))
  (bpp-edit-reset))

(defun bpp-edit-toggle-unknown (arg)
  "Toggle writable status for unknown conditionals.
With optional argument ARG, make them writable iff ARG is positive."
  (interactive "@P")
  (if (or (and (null arg) bpp-unknown-writable)
	  (<= (prefix-numeric-value arg) 0))
      (setq bpp-unknown-writable nil)
    (setq bpp-unknown-writable t))
  (bpp-edit-reset))

(defun bpp-edit-true (symbol face)
  "Select SYMBOL's true FACE used for highlighting taken conditionals."
  (interactive
   (let ((symbol (bpp-choose-symbol)))
     (list symbol
	   (bpp-choose-face "True face"
			    (nth 1 (assoc symbol bpp-edit-list))))))
  (setcar (nthcdr 1 (bpp-edit-list-entry-get-or-create symbol)) face)
  (bpp-edit-reset))

(defun bpp-edit-false (symbol face)
  "Select SYMBOL's false FACE used for highlighting untaken conditionals."
  (interactive
   (let ((symbol (bpp-choose-symbol)))
     (list symbol
	   (bpp-choose-face "False face" 
			    (nth 2 (assoc symbol bpp-edit-list))))))
  (setcar (nthcdr 2 (bpp-edit-list-entry-get-or-create symbol)) face)
  (bpp-edit-reset))

(defun bpp-edit-write (symbol branch)
  "Set which branches of SYMBOL should be writable to BRANCH.
BRANCH should be either nil (false branch), t (true branch) or 'both."
  (interactive (list (bpp-choose-symbol) (bpp-choose-branch)))
  (setcar (nthcdr 3 (bpp-edit-list-entry-get-or-create symbol)) branch)
  (bpp-edit-reset))

(defun bpp-edit-list-entry-get-or-create (symbol)
  ;; Return the entry for SYMBOL in `bpp-edit-list'.
  ;; If it does not exist, create it.
  (let ((entry (assoc symbol bpp-edit-list)))
    (or entry
	(setq entry (list symbol nil nil 'both nil)
	      bpp-edit-list (cons entry bpp-edit-list)))
    entry))

;;; Prompts:

(defun bpp-choose-symbol ()
  ;; Choose a symbol if called from keyboard, otherwise use the one clicked on.
  (if bpp-button-event
      bpp-callback-data
    (completing-read "Symbol: " (mapcar 'list bpp-edit-symbols) nil t)))

(defun bpp-choose-branch ()
  ;; Choose a branch, either nil, t, or both.
  (if bpp-button-event
      (x-popup-menu bpp-button-event
		    (list "Branch" (cons "Branch" bpp-branch-list)))
    (cdr (assoc	(completing-read "Branch: " bpp-branch-list nil t)
		bpp-branch-list))))

(defun bpp-choose-face (prompt default)
  ;; Choose a face from bpp-face-defalt-list.
  ;; PROMPT is what to say to the user.
  ;; DEFAULT is the default face.
  (or (if bpp-button-event
	  (x-popup-menu bpp-button-event
			(list prompt (cons prompt bpp-face-default-list)))
	(let ((name (car (rassq default bpp-face-default-list))))
	  (cdr (assoc (completing-read (if name
					   (concat prompt
						   " (default " name "): ")
					 (concat prompt ": "))
				       bpp-face-default-list nil t)
		      bpp-face-all-list))))
      default))

(defun bpp-choose-default-face (type)
  ;; Choose default face list for screen of TYPE.
  ;; Type must be one of the types defined in `bpp-face-type-list'.
  (interactive (list (if bpp-button-event
			 (x-popup-menu bpp-button-event
				       (list "Screen type"
					     (cons "Screen type"
						   bpp-face-type-list)))
		       (cdr (assoc (completing-read "Screen type: "
						    bpp-face-type-list
						    nil t)
				   bpp-face-type-list)))))
  (cond ((null type))
	((eq type 'light)
	 (if bpp-face-light-list
	     ()
	   (setq bpp-face-light-list
		 (mapcar 'bpp-create-bg-face bpp-face-light-name-list))
	   (setq bpp-face-all-list
		 (append bpp-face-all-list bpp-face-light-list)))
	 (setq bpp-face-type 'light)
	 (setq bpp-face-default-list
	       (append bpp-face-light-list bpp-face-none-list)))
	((eq type 'dark)
	 (if bpp-face-dark-list
	     ()
	   (setq bpp-face-dark-list
		 (mapcar 'bpp-create-bg-face bpp-face-dark-name-list))
	   (setq bpp-face-all-list
		 (append bpp-face-all-list bpp-face-dark-list)))
	 (setq bpp-face-type 'dark)
	 (setq bpp-face-default-list
	       (append bpp-face-dark-list bpp-face-none-list)))
	((eq type 'mono)
	 (setq bpp-face-type 'mono)
	 (setq bpp-face-default-list
	       (append bpp-face-mono-list bpp-face-none-list)))
	(t
	 (setq bpp-face-type 'none)
	 (setq bpp-face-default-list bpp-face-none-list))))

;;; Buttons:

(defun bpp-make-button (name callback &optional data face padding)
  ;; Create a button at point.
  ;; NAME is the name of the button.
  ;; CALLBACK is the function to call when the button is pushed.
  ;; DATA will be made available to CALLBACK
  ;;in the free variable bpp-callback-data.
  ;; FACE means that NAME is the name of a face in `bpp-face-all-list'.
  ;; PADDING means NAME will be right justified at that length.
  (let ((name (format "%s" name))
	from to)
    (cond ((null padding)
	   (setq from (point))
	   (insert name))
	  ((> (length name) padding)
	   (setq from (point))
	   (insert (substring name 0 padding)))
	  (t
	   (insert (make-string (- padding (length name)) ? ))
	   (setq from (point))
	   (insert name)))
    (setq to (point))
    (setq face
	  (if face
	      (let ((check (cdr (assoc name bpp-face-all-list))))
		(if (memq check '(default invisible))
		    'bold
		  check))
	    'bold))
    (add-text-properties from to
			 (append (list 'face face)
				 '(mouse-face highlight)
				 (list 'bpp-callback callback)
				 (if data (list 'bpp-data data))))))

(defun bpp-push-button (event)
  ;; Pushed a BPP button.
  (interactive "@e")
  (set-buffer (window-buffer (posn-window (event-start event))))
  (let ((pos (posn-point (event-start event))))
    (let ((bpp-callback-data (get-text-property pos 'bpp-data))
	  (fun (get-text-property pos 'bpp-callback))
	  (bpp-button-event event))
      (cond (fun
	     (call-interactively (get-text-property pos 'bpp-callback)))
	    ((lookup-key global-map [ down-mouse-2])
	     (call-interactively (lookup-key global-map [ down-mouse-2])))))))

;;; Faces:

(defun bpp-create-bg-face (color)
  ;; Create entry for face with background COLOR.
  (let ((name (intern (concat "bpp " color))))
    (make-face name)
    (set-face-background name color)
    (cons color name)))

(bpp-choose-default-face (if window-system bpp-face-type 'none))

(defun bpp-face-name (face)
  ;; Return the name of FACE from `bpp-face-all-list'.
  (let ((entry (rassq (if face face 'default) bpp-face-all-list)))
    (if entry
	(car entry)
      (format "<%s>" face))))

;;; Utilities:

(defvar bpp-progress-time 0)
;; Last time we issued a progress message.

(defun bpp-progress-message (&rest args)
  ;; Report progress at most once a second.  Take same ARGS as `message'.
  (let ((time (nth 1 (current-time))))
    (if (= time bpp-progress-time)
	()
      (setq bpp-progress-time time)
      (apply 'message args))))

(provide 'bpp)

;;; bpp.el ends here
