unit Bgi;

{ The graphics interface of Graphics Vision.
  Copr. 1997,1998 Matthias K"oppe <mkoeppe@cs.uni-magdeburg.de>
  This library is free software in the sense of the GNU Library GPL;
  see `License Conditions' below.

  $Id: bgi.pp 1.6 1999/02/13 15:57:34 mkoeppe Exp $
 
Info:

 This unit provides the functions of Borland's Graph unit and MKM's
 Gr/MetaGr/MyFonts/VgaMem/GvDriver units. It uses the clipping parameters
 known from MKM's Gr unit. The main target is FPK Pascal, but it can also
 be used with Borland/Turbo Pascal.

 The Graph unit in the Linux RTL has been derived from an early
 version of this unit.

Version:

 As of this version, the following targets are supported.

 * FPK Pascal 0.9 (DOS) with Graph unit

 * FPK Pascal 0.99.5+ (Linux) with libvga and libvgagl (svgalib unit)

     Be sure you have a recent SVGALIB version. You can get it from sunsite.
     Only root can run SVGALIB programs. Use the environment variable
     SVGALIB_DEFAULT_MODE (SVGALIB versions prior to 1.3.0: GSVGAMODE)
     to select the graphics mode; e.g. (in bash):
          export SVGALIB_DEFAULT_MODE=G640x480x256
     See the SVGALIB documentation for more information.

 Large parts have not yet been implemented or tested.

History:

 Date       Version  Who     Comments
 ---------- -------- ------- -------------------------------------
 25-Sep-97  0.1      mkoeppe Initial multi-target version.
 05-Oct-97  0.1.1    mkoeppe Linux: Added mouse use. Improved clipping.
                             Added bitmap functions.
 ??-Oct-97  0.1.2    mkoeppe Fixed screenbuf functions.
 07-Feb-98  0.1.3    mkoeppe Fixed a clipping bug in DOS target.
 12-Apr-98  0.1.4    mkoeppe Linux: Using Michael's re-worked SVGALIB
                             interface; prepared for FPC 0.99.5; removed
                             dependencies.
 06-Jun-98  0.1.5    mkoeppe DOS: Using 800x600x256 as standard mode
                             instead of the maximum mode. Proper colors
                             in all modes.
 06-Feb-99  0.2      mkoeppe Linux: Re-imported some code from graph.pp.
  
Files:

 bgi.pp         This unit.
 tbgi.pp        A very simple test program.
 svgalib.pp     An interface unit for libvga and libvgagl.
 gvfpk.pp       Support unit for FPK Pascal. (DOS target only)
 platform.inc   Platform definitions.

License Conditions:

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
}

interface

{$i platform.inc}

{$ifdef OS_LINUX}
uses Objects, Linux, SvgaLib;
{$endif}
{$ifdef OS_DOS}
{$ifdef PPC_FPC}
uses Objects, Graph, GvFPK;
{$endif}
{$ifdef PPC_BP}
uses Objects, Graph;
{$endif}
{$endif}

{ Graphics constants
}
const
  NormalPut       = 0;
  CopyPut         = 0;
  XORPut          = 1;
  ORPut           = 2;
  ANDPut          = 3;
  NotPut          = 4;
  BackPut         = 8;

  Black           =  0;
  Blue            =  1;
  Green           =  2;
  Cyan            =  3;
  Red             =  4;
  Magenta         =  5;
  Brown           =  6;
  LightGray       =  7;
  DarkGray        =  8;
  LightBlue       =  9;
  LightGreen      = 10;
  LightCyan       = 11;
  LightRed        = 12;
  LightMagenta    = 13;
  Yellow          = 14;
  White           = 15;
  Border          = 16;

  SolidLn         = 0;
  DottedLn        = 1;
  CenterLn        = 2;
  DashedLn        = 3;
  UserBitLn       = 4;

  EmptyFill       = 0;
  SolidFill       = 1;
  LineFill        = 2;
  LtSlashFill     = 3;
  SlashFill       = 4;
  BkSlashFill     = 5;
  LtBkSlashFill   = 6;
  HatchFill       = 7;
  XHatchFill      = 8;
  InterleaveFill  = 9;
  WideDotFill     = 10;
  CloseDotFill    = 11;
  UserFill        = 12;

  NormWidth       = 1;
  ThickWidth      = 3;

const
  LeftText      = 0;
  CenterText    = 1;
  RightText     = 2;
  BottomText    = 0;
  TopText       = 2;
  BaseLine      = 3;
  LeadLine      = 4;

type
  PointType = record
     x,y : integer;
  end;

  ArcCoordsType = record
     x,y : integer;
     xstart,ystart : integer;
     xend,yend : integer;
  end;

{ Retrieving coordinates
}
function GetX: Integer;					{ NO META }
function GetY: Integer;					{ NO META }

{ Pixel-oriented routines
}
procedure PutPixel(X, Y: Integer; Pixel: Word);
function GetPixel(X, Y: Integer): Word;			{ NO META }

{ Line-oriented primitives
}
procedure SetWriteMode(WriteMode: Integer);
procedure LineTo(X, Y: Integer);
procedure LineRel(Dx, Dy: Integer);
procedure MoveTo(X, Y: Integer);
procedure MoveRel(Dx, Dy: Integer);
procedure Line(x1, y1, x2, y2: Integer);
procedure SetLineStyle(LineStyle: Word; Pattern: Word; Thickness: Word);

{ Linearly bounded primitives
}
procedure Rectangle(x1, y1, x2, y2: Integer);
procedure Bar(x1, y1, x2, y2: Integer);
procedure Bar3D(x1, y1, x2, y2: Integer; Depth: Word; Top: Boolean);
procedure DrawPoly(NumPoints: Word; var PolyPoints);
procedure FillPoly(NumPoints: Word; var PolyPoints);
procedure SetFillStyle(Pattern: Word; Color: Word);
{$ifndef OS_LINUX}
procedure SetFillPattern(Pattern: FillPatternType; Color: Word);
{$endif}
procedure FloodFill(X, Y: Integer; Border: Word);

{ Nonlinearly bounded primitives
}
procedure Arc(X, Y: Integer; StAngle, EndAngle, Radius: Word);
{$ifndef OS_LINUX}
procedure GetArcCoords(var ArcCoords: ArcCoordsType);	{ NO META }
{$endif}
procedure Circle(X, Y: Integer; Radius: Word);
procedure Ellipse(X, Y: Integer;
  StAngle, EndAngle: Word; XRadius, YRadius : Word);
procedure FillEllipse(X, Y: Integer; XRadius, YRadius : Word);
procedure SetAspectRatio(Xasp, Yasp: Word);
procedure PieSlice(X, Y: Integer; StAngle, EndAngle, Radius: Word);
procedure Sector(X, Y: Integer;
  StAngle, EndAngle, XRadius, YRadius: Word);

{ Color routines
}
procedure SetBkColor(ColorNum: Word);
procedure SetColor(Color: Word);

{ Bitmap utilities
}
procedure GetImage(x1, y1, x2, y2: Integer; var BitMap);	{ NO META }
procedure PutImage(X, Y: Integer; var BitMap; BitBlt: Word);
function ImageSize(x1, y1, x2, y2: Integer): LongInt;

{ Text routines
}
procedure OutText(TextString: string);
procedure OutTextXY(X, Y: Integer; TextString: string);
procedure SetTextJustify(Horiz, Vert: Word);
procedure SetTextStyle(Font, Direction: Word; CharSize: Word);
procedure SetUserCharSize(MultX, DivX, MultY, DivY: Word);

{ Graph's clipping method
}
procedure SetViewPort(x1, y1, x2, y2: Integer; Clip: Boolean);

{ Graphics Vision functions
}

{ Physical
}
var
  SizeX, SizeY: Word;

function GetMaxX: Word;
function GetMaxY: Word;
function GetMaxColor: LongInt;
function GetBytesPerPixel: Integer;

{ Draw origin and clipping rectangle
}
var
  DrawOrigin: TPoint;
  ClipRect: TRect;
  MetaClipRect: TRect;
  MetaOrigin: TPoint;

procedure SetDrawOrigin(x, y: Integer);
procedure SetDrawOriginP(var P: TPoint);
procedure SetClipRect(x1, y1, x2, y2: Integer);
procedure SetClipRectR(var R: TRect);
Procedure SetMetaOrigin(x,y: Integer);
Procedure SetMetaOriginP(P: TPoint);
Procedure SetMetaClipRect(x1, y1, x2, y2: Integer);
Procedure SetMetaClipRectR(var R: TRect);

{ Meta
}
const
  MetaError: Word = 0;

function GetBuffer(Size: Word): pointer;

{ Font attributes
}
const
  ftNormal          = 0;
  ftBold            = 1;
  ftThin            = 2;
  ftItalic          = 4;

var
  sFont, sColor:Word;
  sCharSpace: Integer;
  sMarker: Char;
  sAttr: Word;
  sHoriz, sVert: Word;

{ Windows-style text metric
}
type
  PTextMetric = ^TTextMetric;
  TTextMetric = record
    tmHeight: Integer;
    tmAscent: Integer;
    tmDescent: Integer;
    tmInternalLeading: Integer;
    tmExternalLeading: Integer;
    tmAveCharWidth: Integer;
    tmMaxCharWidth: Integer;
    tmWeight: Integer;
    tmItalic: Byte;
    tmUnderlined: Byte;
    tmStruckOut: Byte;
    tmFirstChar: Byte;
    tmLastChar: Byte;
    tmDefaultChar: Byte;
    tmBreakChar: Byte;
    tmPitchAndFamily: Byte;
    tmCharSet: Byte;
    tmOverhang: Integer;
    tmDigitizedAspectX: Integer;
    tmDigitizedAspectY: Integer;
  end;

{ Special graphics
}
Procedure HoriLine(x1,y1,x2: Integer);
Procedure VertLine(x1,y1,y2: Integer);
procedure SetTextParams(Font: Word; CharSpace: Integer; Color: Word;
  UseMarker: Boolean);
Procedure FillCircle (xm,ym,r: Integer);
procedure SetKern(Enable: Boolean);
procedure SetMarker(Marker: Char);
function TextWidth(s: string): Integer;
function TextHeight(s: string): Integer;
procedure GetTextMetrics(var Metrics: TTextMetric);
procedure InitFonts;

{ Init/Done
}
{$ifdef LINUX}
const { gr modes match SVGALIB modes }
  gr360x480x256	 = G360x480x256;
  gr640x480x16	 = G640x480x16;
  gr640x480x256	 = G640x480x256;
  gr800x600x256	 = G800x600x256;
  gr1024x768x256 = G1024x768x256;
  gr640x480x32k	 = G640x480x32K;
  gr800x600x32k	 = G800x600x32K;
  gr1024x768x32k = G1024x768x32K;
  gr640x480x64k	 = G640x480x64K;
  gr800x600x64k	 = G800x600x64K;
  gr1024x768x64k = G1024x768x64K;
{$else}
const { gr modes match VESA modes }
  gr640x400x256	 = $100;
  gr640x480x256	 = $101;
  gr800x600x256	 = $103;
  gr1024x768x256 = $105;
  gr640x480x32k	 = $110;
  gr800x600x32k	 = $113;
  gr1024x768x32k = $116;
  gr640x480x64k	 = $111;
  gr800x600x64k	 = $114;
  gr1024x768x64k = $117;
{$endif}

const
  StdGrMode: Integer = gr800x600x256;
var
  GrMode: Integer;

procedure InitVideo; 
procedure DoneVideo;
procedure SetScreenMode(Mode: Integer);

function GetResX: Integer;
function GetResY: Integer;
function GetAspect: Real;

const
  NoGraphics: Boolean = false;

const
  IndirectGraphics: Boolean = false;

{ VgaMem 
}
procedure CopyScreen(x1, y1, x2, y2, x3, y3: Integer);
function CopyImage(Image: pointer): pointer;
function CutImage(x1, y1, x2, y2: Integer): pointer;
procedure FreeImage(Image: pointer);
procedure GetImageExtent(Image: pointer; var Extent: TPoint);
function LoadImage(var S: TStream): pointer;
function MaskedImage(Image: pointer): pointer;
procedure PasteImage(X, Y: Integer; Image: pointer; BitBlt: Word);
procedure StoreImage(var S: TStream; Image: pointer);

{procedure PutIconAnd(x1, y1: Integer; MapMask: Byte; Icon: pointer);
procedure PutIconOr(x1, y1: Integer; MapMask: Byte; Icon: pointer);
procedure PutIconCol(x1, y1: Integer; Col: Integer; Icon: pointer);}

function GetNearestPaletteIndex(RGBColor: LongInt): LongInt;
function GetRGBColor(BgiColor: Byte): LongInt;

{ Storing screen regions
}
type
  TVgaBuf = record
    Bounds: TRect;
    Mem: Word;
    Size: Word;
  end;

const
  pbNone  = 0;
  pbCopy  = 1;
  pbClear = 2;

function PrepBuf(var R: TRect; Action: Word; var Buf: TVgaBuf): Boolean;
procedure EndBufDraw;
procedure ReleaseBuf(var Buf: TVgaBuf);
procedure PasteRect(var R: TRect; var Buf: TVgaBuf);
procedure PasteRectAt(var R: TRect; P: TPoint; var Buf: TVgaBuf);
procedure UpdatePhysicalScreen(var ER: TRect);

type
  PScreenBuf = ^TScreenBuf;
  TScreenBuf = record
    Mode: Word;
    Rect: TRect;
    Size: LongInt;
    Info: LongInt
  end;

function StoreScreen(x1, y1, x2, y2: Integer): PScreenBuf;
procedure FreeScreenBuf(Buf: PScreenBuf);
procedure DrawScreenBufAt(Buf: PScreenBuf; x3, y3: Integer);
procedure DrawScreenBuf(Buf: PScreenBuf);

{ initialisierte Variablen
}
const
  SourcePage: Word = 0;
  DestPage: Word = 0;

{ Retrieves the capabilities for the current mode
}
const
  vmcImage       = 1;
  vmcCopy        = 2;
  vmcSaveRestore = 4;
  vmcBuffer      = 8;
  vmcBackPut	 = 16;

function GetVgaMemCaps: Word;

{$ifdef Linux}
var
  PhysicalScreen, BackScreen, ShadowScreen: PGraphicsContext;
{$endif}

implementation

{ Common routines --- }

const
  BgiColors: array[0..15] of LongInt
    = ($000000, $000080, $008000, $008080,
       $800000, $800080, $808000, $C0C0C0,
       $808080, $0000FF, $00FF00, $00FFFF,
       $FF0000, $FF00FF, $FFFF00, $FFFFFF);

const
  NoDraw : Boolean = false;

type
  PBitmap = ^TBitmap;
  TBitmap = record
	      Width, Height: Integer;
	      Data: record end;
	    end;

function GetRGBColor(BgiColor: Byte): LongInt;
begin
  GetRGBColor := BgiColors[BgiColor]
end;

function CopyImage(Image: pointer): pointer;
var
  C: pointer;
  Extent: TPoint;
  Size: LongInt;
begin
  if Image = nil then begin
    CopyImage := nil;
    Exit
  end;
  GetImageExtent(Image, Extent);
  Size := ImageSize(0, 0, Extent.X - 1, Extent.Y - 1);
  GetMem(C, Size);
  Move(Image^, C^, Size);
  CopyImage := C;
end;

function CutImage(x1, y1, x2, y2: Integer): pointer;
var
  Image: PBitmap;
begin
  GetMem(Image, ImageSize(x1, y1, x2, y2));
  if Image <> nil
    then GetImage(x1, y1, x2, y2, Image^);
  CutImage := Image;
end;

procedure FreeImage(Image: pointer);
var
  P: TPoint;
begin
  if Image <> nil
    then begin
      GetImageExtent(Image, P);
      FreeMem(Image, ImageSize(0, 0, P.x - 1, P.y - 1));
    end;
end;

procedure GetImageExtent(Image: pointer; var Extent: TPoint);
begin
  if Image = nil
    then begin
      Extent.X := 0;
      Extent.Y := 0
    end
    else begin
      Extent.X := PBitmap(Image)^.Width;
      Extent.Y := PBitmap(Image)^.Height
    end;
end;

function LoadImage(var S: TStream): pointer;
begin
  LoadImage := nil
end;

function MaskedImage(Image: pointer): pointer;
begin
  MaskedImage := nil;
end;

procedure PasteImage(X, Y: Integer; Image: pointer; BitBlt: Word);
begin
  if Image <> nil then PutImage(X, Y, Image^, BitBlt)
{  else begin /// debug
     Setfillstyle(solidfill, 5);
     bar(X, Y, X+8, Y+8)
  end;}
end;

procedure StoreImage(var S: TStream; Image: pointer);
begin
end;

{$ifdef OS_DOS}

var
  ColorTable: array[0..15] of LongInt;

var
  DrawDelta: TPoint;

function GetX: Integer;					{ NO META }
begin
  GetX := Graph.GetX - DrawDelta.X
end;

function GetY: Integer;					{ NO META }
begin
  GetY := Graph.GetY - DrawDelta.Y
end;

{ Pixel-oriented routines
}
procedure PutPixel(X, Y: Integer; Pixel: Word);
begin
  Graph.PutPixel(X + DrawDelta.X, Y + DrawDelta.Y, Pixel)
end;

function GetPixel(X, Y: Integer): Word;			{ NO META }
begin
  GetPixel := Graph.GetPixel(X + DrawDelta.X, Y + DrawDelta.Y)
end;

{ Line-oriented primitives
}
procedure SetWriteMode(WriteMode: Integer);
begin
  Graph.SetWriteMode(WriteMode)
end;

procedure LineTo(X, Y: Integer);
begin
  Graph.LineTo(X + DrawDelta.X, Y + DrawDelta.Y)
end;

procedure LineRel(Dx, Dy: Integer);
begin
  Graph.LineRel(Dx, Dy)
end;

procedure MoveTo(X, Y: Integer);
begin
  Graph.MoveTo(X + DrawDelta.X, Y + DrawDelta.Y)
end;

procedure MoveRel(Dx, Dy: Integer);
begin
  Graph.MoveRel(Dx, Dy)
end;

procedure Line(x1, y1, x2, y2: Integer);
begin
  Graph.Line(x1 + DrawDelta.X, y1 + DrawDelta.Y,
    x2 + DrawDelta.X, y2 + DrawDelta.Y)
end;

procedure SetLineStyle(LineStyle: Word; Pattern: Word; Thickness: Word);
begin
  Graph.SetLineStyle(LineStyle, Pattern, Thickness)
end;

{ Linearly bounded primitives
}
procedure Rectangle(x1, y1, x2, y2: Integer);
begin
  Graph.Rectangle(x1 + DrawDelta.X, y1 + DrawDelta.Y,
    x2 + DrawDelta.X, y2 + DrawDelta.Y)
end;

procedure Bar(x1, y1, x2, y2: Integer);
begin
  Graph.Bar(x1 + DrawDelta.X, y1 + DrawDelta.Y,
    x2 + DrawDelta.X, y2 + DrawDelta.Y)
end;

procedure Bar3D(x1, y1, x2, y2: Integer; Depth: Word; Top: Boolean);
begin
  Graph.Bar3D(x1 + DrawDelta.X, y1 + DrawDelta.Y,
    x2 + DrawDelta.X, y2 + DrawDelta.Y, Depth, Top)
end;

procedure DrawPoly(NumPoints: Word; var PolyPoints);
begin
end;

procedure FillPoly(NumPoints: Word; var PolyPoints);
begin
end;

procedure SetFillStyle(Pattern: Word; Color: Word);
begin
  Graph.SetFillStyle(Pattern, ColorTable[Color])
end;

procedure SetFillPattern(Pattern: FillPatternType; Color: Word);
begin
  Graph.SetFillPattern(Pattern, Color)
end;

procedure FloodFill(X, Y: Integer; Border: Word);
begin
end;

{ Nonlinearly bounded primitives
}
procedure Arc(X, Y: Integer; StAngle, EndAngle, Radius: Word);
begin
  Graph.Arc(X + DrawDelta.X, Y + DrawDelta.Y, StAngle, EndAngle, Radius)
end;

procedure GetArcCoords(var ArcCoords: Bgi.ArcCoordsType);	{ NO META }
begin
end;

procedure Circle(X, Y: Integer; Radius: Word);
begin
  Graph.Circle(X + DrawDelta.X, Y + DrawDelta.Y, Radius)
end;

procedure Ellipse(X, Y: Integer;
  StAngle, EndAngle: Word; XRadius, YRadius : Word);
begin
{$ifndef FPK}
  Graph.Ellipse(X + DrawDelta.X, Y + DrawDelta.Y, StAngle, EndAngle,
    XRadius, YRadius)
{$endif}
end;

procedure FillEllipse(X, Y: Integer; XRadius, YRadius : Word);
begin
  Graph.FillEllipse(X + DrawDelta.X, Y + DrawDelta.Y, XRadius, YRadius)
end;

procedure SetAspectRatio(Xasp, Yasp: Word);
begin
  Graph.SetAspectRatio(Xasp, Yasp)
end;

procedure PieSlice(X, Y: Integer; StAngle, EndAngle, Radius: Word);
begin
{$ifndef FPK}
  Graph.PieSlice(X + DrawDelta.X, Y + DrawDelta.Y, StAngle, EndAngle, Radius)
{$endif}
end;

procedure Sector(X, Y: Integer;
  StAngle, EndAngle, XRadius, YRadius: Word);
begin
{$ifndef FPK}
  Graph.Sector(X + DrawDelta.X, Y + DrawDelta.Y,
    StAngle, EndAngle, XRadius, YRadius)
{$endif FPK}
end;

{ Color routines
}
var
  TheColor: Word;

procedure SetBkColor(ColorNum: Word);
begin
end;

procedure SetColor(Color: Word);
begin
  TheColor := Color;
  Graph.SetColor(ColorTable[Color])
end;

{ Bitmap utilities
}
procedure GetImage(x1, y1, x2, y2: Integer; var BitMap);	{ NO META }
begin
  Graph.GetImage(x1 + DrawDelta.X, y1 + DrawDelta.Y,
    x2 + DrawDelta.X, y2 + DrawDelta.Y, Bitmap)
end;

procedure PutImage(X, Y: Integer; var BitMap; BitBlt: Word);
begin
  Graph.PutImage(x + DrawDelta.X, y + DrawDelta.Y, Bitmap,
		 Bitblt and not backput)
end;

{ Graphics Vision
}
function GetNearestPaletteIndex(RGBColor: LongInt): LongInt;
begin
  GetNearestPaletteIndex := RGBColor
end;

procedure SetColors;
var
  i: Integer;
begin
   { Yes this looks silly but in the new graph unit,
    the color names are variables }
   ColorTable[0] := Graph.Black;
   ColorTable[1] := Graph.Blue;
   ColorTable[2] := Graph.Green;
   ColorTable[3] := Graph.cyan;
   ColorTable[4] := Graph.red;
   ColorTable[5] := Graph.magenta;
   ColorTable[6] := Graph.brown;
   ColorTable[7] := Graph.lightgray;
   ColorTable[8] := Graph.darkgray;
   ColorTable[9] := Graph.lightblue;
   ColorTable[10] := Graph.lightgreen;
   ColorTable[11] := Graph.lightcyan;
   ColorTable[12] := Graph.lightred;
   ColorTable[13] := Graph.lightmagenta;
   ColorTable[14] := Graph.yellow;
   ColorTable[15] := Graph.white;
end;
      
{  case GetMaxColor of
    255:
	begin
	  for i := 0 to 15 do
	  begin
	    ColorTable[i] := i;
	    SetRGBPalette(i, (BgiColors[i] shr 16),
			  ((BgiColors[i] shr 8) and 255),
			  (BgiColors[i] and 255));
	  end;
	end;
    32767:
	  begin
	    for i := 0 to 15 do
	    begin
	      ColorTable[i] := BgiColors[i];
	    end
	  end;
    65535:
	  begin
	    for i := 0 to 15 do
	    begin
	      ColorTable[i] := BgiColors[i];
	    end;
	  end;
  end;
end;}

procedure InitVideo;
var
  d, m: Integer;
begin
{  d := Detect;}
  if NoGraphics
  then begin
    SizeX := 640;
    SizeY := 480
  end
  else begin
     d := $ff;
     m := StdGrMode;
     InitGraph(d, m, '');
     SizeX := Graph.GetMaxX + 1;
     SizeY := Graph.GetMaxY + 1
  end;
{$ifdef PPC_FPC}
  SetMouseArea(0, 0, SizeX, SizeY);
{$endif}
  SetColors;
end;

procedure DoneVideo;
begin
  CloseGraph;
end;

procedure InitFonts;
begin
end;

procedure SetScreenMode(Mode: Integer);
begin
  SetGraphMode(Mode);
  SetColors;
end;

procedure SetDelta;
begin
  if ClipRect.Empty
  then begin
    DrawDelta.X := 10000;
    DrawDelta.Y := 10000;
  end
  else begin
    DrawDelta.X := DrawOrigin.X - ClipRect.A.x;
    DrawDelta.y := DrawOrigin.y - ClipRect.A.y
  end
end;

procedure SetDrawOrigin(x, y: Integer);
begin
  DrawOrigin.x := x;
  DrawOrigin.y := y;
  SetDelta;
end;

procedure SetDrawOriginP(var P: TPoint);
begin
  SetDrawOrigin(P.x, P.y)
end;

procedure SetClipRect(x1, y1, x2, y2: Integer);
begin
  Cliprect.Assign(x1, y1, x2, y2);
  Graph.SetViewPort(x1, y1, x2 - 1, y2 - 1, ClipOn);
  SetDelta
end;

procedure SetClipRectR(var R: TRect);
begin
  SetClipRect(R.A.X, R.A.Y, R.B.X, R.B.Y);
end;

procedure SetMetaOrigin(x, y: Integer);
begin
  MetaOrigin.x := x;
  MetaOrigin.y := y
end;

procedure SetMetaOriginP(P: TPoint);
begin
  SetMetaOrigin(P.x, P.y)
end;

procedure SetMetaClipRect(x1, y1, x2, y2: Integer);
begin
  MetaCliprect.Assign(x1, y1, x2, y2)
end;

procedure SetMetaClipRectR(var R: TRect);
begin
  MetaCliprect := R
end;

function GetBuffer(Size: Word): pointer;
begin
  { No metafiling available. }
  GetBuffer := nil
end;

Procedure HoriLine(x1,y1,x2: Integer);
begin
  Line(x1, y1, x2, y1)
end;

Procedure VertLine(x1,y1,y2: Integer);
begin
  Line(x1, y1, x1, y2)
end;

procedure FillCircle(xm, ym, r: Integer);
begin
  FillEllipse(xm, ym, r, r)
end;

{ Text routines
}

const
  DoUseMarker: Boolean = true;
  TheMarker: Char = '~';
  TextColor: Word = $0f0f;

procedure OutText(TextString: string);
begin
  {Graph.OutText(TextString)}
  OutTextXY(GetX, GetY, TextString)
end;

procedure OutTextXY(X, Y: Integer; TextString: string);
var
  P, Q: PChar;
  i: Integer;
  col: Boolean;
begin
  if TextString='' then Exit;
  if (not DoUseMarker) or (Pos(TheMarker, TextString) = 0)
  then begin
    Graph.SetColor(ColorTable[Lo(TextColor)]);
    Graph.OutTextXY(X + DrawDelta.X, y + DrawDelta.Y, TextString);
    Graph.SetColor(ColorTable[TheColor])
  end
  else begin
    Graph.SetTextJustify(LeftText, sVert);
    case sHoriz of
      LeftText:
	;
      CenterText:
	Dec(x, TextWidth(TextString) div 2);
      RightText:
	Dec(x, TextWidth(TextString));
    end;
    Graph.MoveTo(X + DrawDelta.X, Y + DrawDelta.Y);
    P := @TextString[1]; Q := P;
    col := false;
    Graph.SetColor(ColorTable[Lo(TextColor)]);
    For i := 1 to Length(TextString) do
    begin
      If Q[0] = TheMarker
      then begin
	If col
	then Graph.SetColor(ColorTable[Hi(TextColor)])
	else Graph.SetColor(ColorTable[Lo(TextColor)]);

	If Q <> P
	then begin
	  (P - 1)[0] := Chr(Q - P);
	  Graph.OutText(PString(pointer(P-1))^);
	end;
	col := not col;
	P := Q + 1
      end;
      Inc(Q)
    end;
    If col
    then Graph.SetColor(ColorTable[Hi(TextColor)])
    else Graph.SetColor(ColorTable[Lo(TextColor)]);
    If Q <> P
    then begin
      (P - 1)[0] := Chr(Q - P);
      Graph.OutText(PString(pointer(P-1))^);
    end;
    Graph.SetColor(ColorTable[TheColor]);
    Graph.SetTextJustify(sHoriz, sVert)
  end
end;

procedure SetTextJustify(Horiz, Vert: Word);
begin
  sHoriz := Horiz; sVert := Vert;
  Graph.SetTextJustify(Horiz, Vert)
end;

procedure SetTextStyle(Font, Direction: Word; CharSize: Word);
begin
  Graph.SetTextStyle(Font, Direction, CharSize);
end;

procedure SetUserCharSize(MultX, DivX, MultY, DivY: Word);
begin
  Graph.SetUserCharSize(MultX, DivX, MultY, DivY)
end;

procedure SetKern(Enable: Boolean);
begin
end;

procedure SetMarker(Marker: Char);
begin
  TheMarker := Marker
end;

function TextWidth(s: string): Integer;
var
  i: Integer;
begin
  if DoUseMarker
  then begin
    For i := Length(s) downto 1 do
      If s[i] = TheMarker then Delete(s, i, 1);
    If s = ''
    then TextWidth := 0
    else TextWidth := Graph.TextWidth(s)
  end
  else TextWidth := Graph.TextWidth(s)
end;

function TextHeight(s: string): Integer;
begin
  TextHeight := Graph.TextHeight(s)
end;

procedure SetTextParams(Font: Word; CharSpace: Integer; Color: Word;
  UseMarker: Boolean);
begin
  sFont := Font; sColor := Color; sCharSpace := CharSpace;
  TextColor := Color;
  DoUseMarker := UseMarker
end;

function GetResX: Integer;
begin
  GetResX := 96;
end; { GetResX }

function GetResY: Integer;
begin
  GetResY := 96
end; { GetResY }

function GetAspect: Real;
begin
  GetAspect := 1.0
end;

procedure SetViewPort(x1, y1, x2, y2: Integer; Clip: Boolean);
begin
  SetDrawOrigin(x1, y1);
  if Clip then SetClipRect(x1, y1, x2+1, y2+1)
  else SetClipRect(0, 0, SizeX, SizeY)
end;

type
  TImage = record
  end;

{ Restspeicherverwaltung
}
procedure InitVgaMem; begin end;
procedure DoneVgaMem; begin end;
function GetVgaMem(Size: Word): Word; begin GetVgaMem := 0 end;
procedure FreeVgaMem(P, Size: Word); begin end;

{ Restspeicherpufferung
}
function PrepBuf(var R: Objects.TRect; Action: Word; var Buf: TVgaBuf): Boolean;
begin PrepBuf := false end;
procedure EndBufDraw; begin end;
procedure ReleaseBuf(var Buf: TVgaBuf); begin end;
procedure PasteRect(var R: Objects.TRect; var Buf: TVgaBuf); begin end;
procedure PasteRectAt(var R: Objects.TRect; P: Objects.TPoint; var Buf: TVgaBuf); begin end;
procedure UpdatePhysicalScreen(var ER: TRect); begin end;

{ Kopierroutinen
}
function GetSize(x1, y1, x2, y2: Integer): Word; begin GetSize := 0 end;
function GetBPL(x1, x2: Integer): Word; begin GetBPL := 0 end;
procedure SaveScreen(x1, y1, x2, y2: Integer; Addr: Word); begin end;
procedure RestoreScreen(x1, y1, x2, y2, x3, y3: Integer; Addr: Word); begin end;

procedure CopyScreen(x1, y1, x2, y2, x3, y3: Integer);
begin
end;

{ Icon-Darstellung
}
{procedure PutIconAnd16(x1, y1: Integer; MapMask: Byte; Icon: Pointer); begin end;
procedure PutIconOr16(x1, y1: Integer; MapMask: Byte; Icon: Pointer); begin end;
}

{ Low-level routines
}
function ImageSize(x1, y1, x2, y2: Integer): LongInt;
begin
  ImageSize := Graph.ImageSize(x1, y1, x2, y2)
end;

{ Storing screen regions
}

function StoreScreen(x1, y1, x2, y2: Integer): PScreenBuf;
begin StoreScreen := nil end;
procedure FreeScreenBuf(Buf: PScreenBuf); begin end;
procedure DrawScreenBufAt(Buf: PScreenBuf; x3, y3: Integer); begin end;
procedure DrawScreenBuf(Buf: PScreenBuf); begin end;

function GetVgaMemCaps: Word;
begin
  GetVgaMemCaps := 0
end;

{ Old-style functions
}
procedure InitVGAMan; begin end;
procedure DoneVGAMan; begin end;
function GetSize16(x1, y1, x2, y2: Integer): Word; begin GetSize16 := 0 end;
function GetBPL16(x1, x2: Integer): Word; begin GetBPL16 := 0 end;
procedure SaveScreen16(x1, y1, x2, y2: Integer; Addr: Word); begin end;
procedure RestoreScreen16(x1, y1, x2, y2, x3, y3: Integer; Addr: Word); begin end;
procedure CopyScreen16(x1, y1, x2, y2, x3, y3: Integer); begin end;
procedure CopyMem16(Source, Dest, Count: Word); begin end;


procedure GetTextMetrics(var Metrics: TTextMetric);
begin
  with Metrics do
  begin
    tmHeight := 8;
    tmAscent := 8;
    tmDescent := 0;
    tmInternalLeading := 0;
    tmExternalLeading := 0;
    tmAveCharWidth := 8;
    tmMaxCharWidth := 8;
    tmWeight := 700;
    tmItalic := 0;
    tmUnderlined := 0;
    tmStruckOut := 0;
    tmFirstChar := 0;
    tmLastChar := 255;
    tmDefaultChar := 32;
    tmBreakChar := 32;
    tmPitchAndFamily := 0;
    tmCharSet := 0;
    tmOverhang := 0;
    tmDigitizedAspectX := 100;
    tmDigitizedAspectY := 100
  end;
end;

function GetMaxX: Word;
begin
  GetMaxX := SizeX - 1;
end; { GetMaxX }

function GetMaxY: Word;
begin
  GetMaxY := SizeY - 1;
end; { GetMaxY }

function GetMaxColor: LongInt;
begin
  GetMaxColor := Graph.GetMaxColor;
end; { GetMaxColor }

function GetBytesPerPixel: Integer;
var
   i : Integer;
   n : LongInt;
begin
   n := GetMaxColor + 1;
   for i := 1 to 32 do begin
      n := n shr 1;
      if n <= 1 then Break;
   end;
   GetBytesPerPixel := (i + 7) div 8;
end; { GetBytesPerPixel }

{$endif OS_DOS}

{ ---------------------------------------------------------------- }

{$ifdef OS_Linux}

var
  DrawDelta: TPoint;
  CurX, CurY: Integer;
  TheColor, TheFillColor: LongInt;
  ColorTable: array[0..15] of LongInt;

const
  DoUseMarker: Boolean = true;
  TheMarker: Char      = '~';
  TextColor: LongInt   = 15;
  MarkColor: LongInt   = 15;
  BackColor: LongInt   = 0;
  FontWidth: Integer   = 8;
  FontHeight: Integer  = 8;
var
  FontBuf: pointer;

function GetX: Integer;					{ NO META }
begin
  GetX := CurX - DrawDelta.X
end;

function GetY: Integer;					{ NO META }
begin
  GetY := CurY - DrawDelta.Y
end;

function GetBytesPerPixel: Integer;
begin
   { We are interested in the screen that is
    directly connected to bitblitting. }
   GetBytesPerPixel := ShadowScreen^.BytesPerPixel;
end;   

{ Pixel-oriented routines
}
procedure PutPixel(X, Y: Integer; Pixel: Word);
begin
  if not NoDraw
    then gl_setpixel(X + DrawDelta.X, Y + DrawDelta.Y, Pixel)
end;

function GetPixel(X, Y: Integer): Word;			{ NO META }
begin
  if NoDraw
    then GetPixel := 0
    else GetPixel := gl_getpixel(X + DrawDelta.X, Y + DrawDelta.Y)
end;

{ Line-oriented primitives
}
procedure SetWriteMode(WriteMode: Integer);
begin
end;

procedure LineTo(X, Y: Integer);
begin
  if not NoDraw
    then gl_line(CurX, CurY, X + DrawDelta.X, Y + DrawDelta.Y, TheColor);
  CurX := X + DrawDelta.X;
  CurY := Y + DrawDelta.Y
end;

procedure LineRel(Dx, Dy: Integer);
begin
  if not NoDraw
    then gl_line(CurX, CurY, CurX + Dx, CurY + Dy, TheColor);
  CurX := CurX + Dx;
  CurY := CurY + Dy
end;

procedure MoveTo(X, Y: Integer);
begin
  CurX := X + DrawDelta.X;
  CurY := Y + DrawDelta.Y
end;

procedure MoveRel(Dx, Dy: Integer);
begin
  CurX := CurX + Dx;
  CurY := CurY + Dy
end;

procedure Line(x1, y1, x2, y2: Integer);
begin
  if not NoDraw
    then gl_line(x1 + DrawDelta.X, y1 + DrawDelta.Y,
		 x2 + DrawDelta.X, y2 + DrawDelta.Y, TheColor)
end;

procedure SetLineStyle(LineStyle: Word; Pattern: Word; Thickness: Word);
begin
end;

{ Linearly bounded primitives
}
procedure Rectangle(x1, y1, x2, y2: Integer);
begin
  MoveTo(x1, y1);
  LineTo(x2, y1);
  LineTo(x2, y2);
  LineTo(x1, y2);
  LineTo(x1, y1)
end;

procedure Bar(x1, y1, x2, y2: Integer);
var
  R: TRect;
begin
  if not NoDraw
    then begin
      R.Assign(x1 + DrawDelta.X, y1 + DrawDelta.Y,
	       x2 + DrawDelta.X + 1, y2 + DrawDelta.Y + 1);
      R.Intersect(ClipRect);
      if not R.Empty
	then gl_fillbox(R.A.X, R.A.Y,
			R.B.X - R.A.X, R.B.Y - R.A.Y, TheFillColor)
    end;
end;

procedure Bar3D(x1, y1, x2, y2: Integer; Depth: Word; Top: Boolean);
begin
  Bar(x1,y1,x2,y2);
  Rectangle(x1,y1,x2,y2);
  if top then begin
     Moveto(x1,y1);
     Lineto(x1+depth,y1-depth);
     Lineto(x2+depth,y1-depth);
     Lineto(x2,y1);
  end;
  Moveto(x2+depth,y1-depth);
  Lineto(x2+depth,y2-depth);
  Lineto(x2,y2);
end;

procedure DrawPoly(NumPoints: Word; var PolyPoints);
type
   ppointtype = ^pointtype;
var
   i : longint;
begin
   line(ppointtype(@polypoints)[NumPoints-1].x,
        ppointtype(@polypoints)[NumPoints-1].y,
        ppointtype(@polypoints)[0].x,
        ppointtype(@polypoints)[0].y);
   for i:=0 to NumPoints-2 do
     line(ppointtype(@polypoints)[i].x,
          ppointtype(@polypoints)[i].y,
          ppointtype(@polypoints)[i+1].x,
          ppointtype(@polypoints)[i+1].y);
end;

procedure FillPoly(NumPoints: Word; var PolyPoints);
begin
  DrawPoly (NumPoints,PolyPoints);
end;

procedure SetFillStyle(Pattern: Word; Color: Word);
begin
  TheFillColor := ColorTable[Color]
end;

procedure FloodFill(X, Y: Integer; Border: Word);
begin
end;

{ Nonlinearly bounded primitives
}
Var LastArcCoords : ArcCoordsType;


procedure SetArcCoords (X,y,xradius,yradius,Stangle,endangle : integer);   
begin
  LastArcCoords.X:=X;
  LastArccOords.y:=y;
  Lastarccoords.xstart:=x+round(xradius*cos(stangle*pi/180));
  Lastarccoords.ystart:=y-round(yradius*sin(stangle*pi/180));
  LastArccoords.xend:=x+round(xradius*cos(endangle*pi/180));
  LastArccoords.yend:=y-round(yradius*sin(endangle*pi/180));
end;

procedure GetArcCoords(var ArcCoords: ArcCoordsType);   
begin
  ArcCoords:=LastArcCoords;
end;

procedure Arc(X, Y: Integer; StAngle, EndAngle, Radius: Word);
begin
end;

procedure Circle(X, Y: Integer; Radius: Word);
begin
  if not NoDraw
    then gl_circle(X + DrawDelta.X, Y + DrawDelta.Y, Radius, TheColor)
end;

procedure Ellipse(X, Y: Integer;
  StAngle, EndAngle: Word; XRadius, YRadius : Word);
begin
end;

procedure FillEllipse(X, Y: Integer; XRadius, YRadius : Word);
begin
  Bar(X - XRadius, Y - YRadius, X + XRadius, Y + YRadius);
  Rectangle(X - XRadius, Y - YRadius, X + XRadius, Y + YRadius);
end;

procedure SetAspectRatio(Xasp, Yasp: Word);
begin
end;

procedure PieSlice(X, Y: Integer; StAngle, EndAngle, Radius: Word);
Begin
 sector (x,y,stangle,endangle,radius,radius);
end;

procedure Sector(X, Y: Integer;
  StAngle, EndAngle, XRadius, YRadius: Word);
begin
end;

{ Color routines
}

procedure SetBkColor(ColorNum: Word);
begin
  BackColor := ColorTable[ColorNum];
end;

procedure SetColor(Color: Word);
begin
  TheColor := ColorTable[Color];
end;

{ Bitmap utilities
}

procedure GetImage(x1, y1, x2, y2: Integer; var BitMap);	{ NO META }
var
  SaveClipRect: TRect;
begin
  with TBitmap(Bitmap) do
  begin
    Width := x2 - x1 + 1;
    Height := y2 - y1 + 1;
    if not NoGraphics
      then begin
	SaveClipRect := ClipRect;
	SetClipRect(0, 0, SizeX, SizeY);
	gl_getbox(x1 + DrawDelta.X, y1 + DrawDelta.Y,
		  x2 - x1 + 1, y2 - y1 + 1, @Data);
	SetClipRectR(SaveClipRect);
      end;
  end;
end;

procedure PutImage(X, Y: Integer; var BitMap; BitBlt: Word);
var
  R: TRect;
  SaveClipRect: TRect;
begin
  if not NoDraw then
    with TBitmap(Bitmap) do
    begin
      {gl_putbox(x + DrawDelta.X, y + DrawDelta.Y, Width, Height, @Data)}
      R.Assign(X + DrawDelta.X, Y + DrawDelta.Y,
	       X + DrawDelta.X + Width, Y + DrawDelta.Y + Height);
      R.Intersect(ClipRect);
      if not R.Empty
	then begin
	  if BitBlt and BackPut <> 0 then begin
	    gl_putboxmask(X + DrawDelta.X, Y + DrawDelta.Y, Width, Height, @Data);
	  end
	  else begin
	    SaveClipRect := ClipRect;
	    SetClipRect(0, 0, SizeX, SizeY);
	    gl_putboxpart(R.A.X, R.A.Y,
			  R.B.X - R.A.X, R.B.Y - R.A.Y,
			  Width, Height,
			  @Data,
			  R.A.X - X - DrawDelta.X, R.A.Y - Y - DrawDelta.Y);
	    SetClipRectR(SaveClipRect);
	  end;
	end;
    end;
end; { PutImage }

function ImageSize(x1, y1, x2, y2: Integer): LongInt;
begin
  if NoGraphics
    then ImageSize := SizeOf(TBitmap)
    else ImageSize := SizeOf(TBitmap)
      + LongInt(x2 - x1 + 1) * LongInt(y2 - y1 + 1) * BackScreen^.BytesPerPixel;
end;

{ Graphics Vision
}

function GetNearestPaletteIndex(RGBColor: LongInt): LongInt;
var
  R, G, B, v: Integer;
begin
  R := (RGBColor shr 16) and 255;
  G := (RGBColor shr 8) and 255;
  B := RGBColor and 255;
  if PhysicalScreen^.colors = 16 then begin
    { note: this is just adhockery }
    v := 0;
    if abs(R-G)+abs(R-B) < 30 then begin
      { gray }
      if R + G + B < 80 then v := 0 else
	if R + G + B < 400 then v := 8 else
	  if R + G + B < 650 then v := 7 else
	    v := 15;
    end
    else begin
      if R >= 64 then v := 4;
      if G >= 64 then v := v or 2;
      if B >= 64 then v := v or 1;
      if R + G + B > 340 then v := v or 8;
    end;
    GetNearestPaletteIndex := v;
  end
  else GetNearestPaletteIndex := gl_rgbcolor(R, G, B);
end;

procedure SetColors;
var
  i: Integer;
begin
  if (PhysicalScreen^.colors = 16) then
    for i := 0 to 15 do ColorTable[i] := i
  else
    for i:=0 to 15 do
      ColorTable[i] := gl_rgbcolor((BgiColors[i] shr 16),
				   ((BgiColors[i] shr 8) and 255),
				   (BgiColors[i] and 255))
end;

procedure setpalette16;
var
  i, r, g, b: Integer;
begin
  for i := 0 to 15 do begin
    r := (BgiColors[i] shr 16) shr 2;
    g := ((BgiColors[i] shr 8) and 255) shr 2;
    b := (BgiColors[i] and 255) shr 2;
    gl_setpalettecolor(i, r, g, b);
  end;
end;

procedure DoSetScreenMode(Mode: Integer);
var
  ModeInfo: pvga_modeinfo;
begin
  GrMode := Mode;
  if (not vga_hasmode(GrMode))
    then begin
      GrMode := G640x480x16;
      if (not vga_hasmode(GrMode))
	then begin
	  WriteLn('BGI: Mode not available.');
	  Halt(1)
	end
    end;
  ModeInfo := vga_getmodeinfo(GrMode);
  IndirectGraphics := (ModeInfo^.colors = 16) or (ModeInfo^.flags and IS_MODEX <> 0);
  { We always want a back screen (for buffering). }
  { Create virtual screen }
  gl_setcontextvgavirtual(GrMode);
  BackScreen := gl_allocatecontext;
  gl_getcontext(BackScreen);
  if IndirectGraphics then begin
    gl_setcontextvgavirtual(GrMode);
    ShadowScreen := gl_allocatecontext;
    gl_getcontext(ShadowScreen);
  end;
  vga_setmousesupport(1);
  vga_setmode(GrMode);
  gl_setcontextvga(GrMode);  { Physical screen context. }
  PhysicalScreen := gl_allocatecontext;
  gl_getcontext(PhysicalScreen);
  if not IndirectGraphics
    then ShadowScreen := PhysicalScreen;
  gl_setcontext(ShadowScreen);
  if (PhysicalScreen^.colors = 16) then setpalette16;
  if (PhysicalScreen^.colors = 256) then gl_setrgbpalette;
  SetColors;
  SizeX := PhysicalScreen^.Width;
  SizeY := PhysicalScreen^.Height;
  NoDraw := false;
end;

procedure InitVideo;
begin
  if NoGraphics
  then begin
    SizeX := 640;
    SizeY := 480
  end
  else DoSetScreenMode(StdGrMode)
end;

procedure DoneVideo;
begin
  if not NoGraphics
    then begin
      {gl_freecontext(BackScreen);}
	{FIXME: This makes problems with svgalib 1.3.0}
      vga_setmode(GTEXT)
    end
end;

procedure SetScreenMode(Mode: Integer);
begin
  if not NoGraphics then begin
    DoneVideo;
    DoSetScreenMode(Mode);
  end;
end;

function GetMaxX: Word;
begin
  GetMaxX := SizeX - 1;
end; { GetMaxX }

function GetMaxY: Word;
begin
  GetMaxY := SizeY - 1;
end; { GetMaxY }

function GetMaxColor: LongInt;
begin
  GetMaxColor := PhysicalScreen^.colors - 1;
end; { GetMaxColor }

procedure SetDelta;
begin
  if ClipRect.Empty
  then begin
    DrawDelta.X := 10000;
    DrawDelta.Y := 10000;
  end
  else begin
    DrawDelta.X := DrawOrigin.X;
    DrawDelta.y := DrawOrigin.y
  end
end;

procedure SetDrawOrigin(x, y: Integer);
begin
  DrawOrigin.x := x;
  DrawOrigin.y := y;
  SetDelta;
end;

procedure SetDrawOriginP(var P: TPoint);
begin
  SetDrawOrigin(P.x, P.y)
end;

procedure SetClipRect(x1, y1, x2, y2: Integer);
begin
  Cliprect.Assign(x1, y1, x2, y2);
  if not NoGraphics
    then begin
      if ClipRect.Empty
	then NoDraw := true
      else begin
	NoDraw := false;
	gl_setclippingwindow(x1, y1, x2 - 1, y2 - 1);
      end;
    end;
  SetDelta
end;

procedure SetClipRectR(var R: TRect);
begin
  SetClipRect(R.A.X, R.A.Y, R.B.X, R.B.Y);
end;

procedure SetMetaOrigin(x, y: Integer);
begin
  MetaOrigin.x := x;
  MetaOrigin.y := y
end;

procedure SetMetaOriginP(P: TPoint);
begin
  SetMetaOrigin(P.x, P.y)
end;

procedure SetMetaClipRect(x1, y1, x2, y2: Integer);
begin
  MetaCliprect.Assign(x1, y1, x2, y2)
end;

procedure SetMetaClipRectR(var R: TRect);
begin
  MetaCliprect := R
end;

function GetBuffer(Size: Word): pointer;
begin
  { No metafiling available. }
  GetBuffer := nil
end;

Procedure HoriLine(x1,y1,x2: Integer);
begin
  Line(x1, y1, x2, y1)
end;

Procedure VertLine(x1,y1,y2: Integer);
begin
  Line(x1, y1, x1, y2)
end;

procedure FillCircle(xm, ym, r: Integer);
begin
  FillEllipse(xm, ym, r, r)
end;

{ Text routines
}

function FixCol(Col: Byte): Byte;
{ SVGALIB cannot write black characters... }
begin
  if Col=0 then FixCol := 1 else FixCol := Col
end; { FixCol }

procedure OutText(TextString: string);
begin
  OutTextXY(GetX, GetY, TextString)
end;

procedure OutTextXY(X, Y: Integer; TextString: string);
var
  P, Q: PChar;
  i: Integer;
  col: Boolean;
begin
  if NoDraw or (TextString='') then Exit;
  gl_setwritemode(FONT_COMPRESSED + WRITEMODE_MASKED);
  case sHoriz of
    CenterText : Dec(x, TextWidth(TextString) div 2);
    RightText  : Dec(x, TextWidth(TextString));
  end; { case }
  case sVert of
    CenterText : Dec(y, TextHeight(TextString) div 2);
    BottomText, BaseLine : Dec(y, TextHeight(TextString));
  end; { case }
  MoveTo(X, Y);
  P := @TextString[1]; Q := P;
  col := false;
  gl_setfontcolors(BackColor, TextColor);
  For i := 1 to Length(TextString) do
  begin
    If (Q[0] = TheMarker) and DoUseMarker
      then begin
	If col then gl_setfontcolors(BackColor, MarkColor)
	else gl_setfontcolors(BackColor, TextColor);
	If Q <> P then begin
	  gl_writen(CurX, CurY, Q-P, P);
	  MoveRel(FontWidth * (Q-P), 0)
	end;
	col := not col;
	P := Q + 1
      end;
    {Inc(Q)} Q := Q + 1
  end;
  If col then gl_setfontcolors(BackColor, MarkColor)
  else gl_setfontcolors(BackColor, TextColor);
  If Q <> P then begin
    gl_writen(CurX, CurY, Q-P, P);
    MoveRel(FontWidth * (Q-P), 0)
  end
end;

{$ifdef ooo}
procedure PutIconAnd(x1, y1: Integer; MapMask: Byte; Icon: pointer);
begin
  PutIconCol(x1, y1, MapMask, Icon);
end; { PutIconAnd }

procedure PutIconOr(x1, y1: Integer; MapMask: Byte; Icon: pointer);
begin
  PutIconCol(x1, y1, MapMask, Icon);
end; { PutIconOr }

procedure PutIconCol(x1, y1: Integer; Col: Integer; Icon: pointer);
const
  C: Char = #1;
type
  PWord	= ^Word;
begin 
  if NoDraw or (Icon=nil) then Exit;
  gl_setfont(16, PWord(Icon)^, PChar(Icon)+2 - PWord(Icon)^*2);
  gl_setwritemode(FONT_COMPRESSED + WRITEMODE_MASKED);
  gl_setfontcolors(0, ColorTable[FixCol(Col)]);
  gl_writen(x1, y1, 1, @C);
  gl_setfont(FontWidth, FontHeight, FontBuf);
end; { PutIconCol }
{$endif}

procedure SetTextJustify(Horiz, Vert: Word);
begin
  sHoriz := Horiz; sVert := Vert;
end;

procedure SetTextStyle(Font, Direction: Word; CharSize: Word);
begin
end;

procedure SetUserCharSize(MultX, DivX, MultY, DivY: Word);
begin
end;

procedure SetKern(Enable: Boolean);
begin
end;

procedure SetMarker(Marker: Char);
begin
  TheMarker := Marker
end;

function TextWidth(s: string): Integer;
var
  i: Integer;
begin
  if DoUseMarker
  then begin
    For i := Length(s) downto 1 do
      If s[i] = TheMarker then Delete(s, i, 1);
    If s = ''
    then TextWidth := 0
    else TextWidth := Length(s) * FontWidth
  end
  else TextWidth := Length(s) * FontWidth
end;

function TextHeight(s: string): Integer;
begin
  TextHeight := FontHeight
end;

procedure SetTextParams(Font: Word; CharSpace: Integer; Color: Word;
  UseMarker: Boolean);
begin
  sColor := Color; sCharSpace := CharSpace; sFont := Font;
  if not NoDraw then begin
    TextColor := ColorTable[FixCol(Color and 15)];
    MarkColor := ColorTable[FixCol((Color shr 8) and 15)];
    DoUseMarker := UseMarker;
    gl_setfont(FontWidth, FontHeight, FontBuf);
  end
end;

function GetResX: Integer;
begin
  GetResX := 96;
end; { GetResX }

function GetResY: Integer;
begin
  GetResY := 96
end; { GetResY }

function GetAspect: Real;
begin
  GetAspect := 1.0
end; { GetAspect }

procedure SetViewPort(x1, y1, x2, y2: Integer; Clip: Boolean);
begin
  SetDrawOrigin(x1, y1);
  if Clip then SetClipRect(x1, y1, x2+1, y2+1)
  else SetClipRect(0, 0, SizeX, SizeY)
end;

{ VGAMEM }

type
  TImage = record
  end;

procedure CopyScreen(x1, y1, x2, y2, x3, y3: Integer);
var
  R: TRect;
begin
  if not NoDraw and (x2 > x1) and (y2 > y1) then begin
    gl_copyboxfromcontext(ShadowScreen^, x1, y1, x2 - x1, y2 - y1, x3, y3);
    if IndirectGraphics then begin
      R.Assign(x3, y3, x3+x2-x1, y3+y2-y1);
      UpdatePhysicalScreen(R);
    end
  end
end;

{ BGI-like Image routines
}

{ Storing screen regions
}
function PrepBuf(var R: TRect; Action: Word; var Buf: TVgaBuf): Boolean;
begin
  if BackScreen <> nil
    then begin
      Buf.Bounds := R;
      gl_setcontext(BackScreen);
      gl_disableclipping;
      if IndirectGraphics then begin
	gl_fillbox(R.A.X and not 7, R.A.Y,
		   ((R.B.X - 1) or 7) - (R.A.X and not 7) + 1, R.B.Y - R.A.Y, 13);
      end;
      case Action of
	pbCopy: gl_copyboxfromcontext(ShadowScreen^,
				      R.A.X, R.A.Y, R.B.X - R.A.X, R.B.Y - R.A.Y,
				      R.A.X, R.A.Y);
	pbClear: gl_fillbox(R.A.X, R.A.Y, R.B.X - R.A.X, R.B.Y - R.A.Y, 0);
      end;
      PrepBuf := true;
      SetDrawOrigin(0, 0);
      SetClipRectR(R);
    end
    else PrepBuf := false
end; { PrepBuf }

procedure EndBufDraw;
begin
  if not NoGraphics
    then begin
      gl_setcontext(ShadowScreen);
      noDraw := false;
    end;
end; { EndBufDraw }

procedure ReleaseBuf(var Buf: TVgaBuf);
begin
end; { ReleaseBuf }

procedure PasteRect(var R: TRect; var Buf: TVgaBuf);
begin
  PasteRectAt(R, R.A, Buf);
end; { PasteRect }

procedure PasteRectAt(var R: TRect; P: TPoint; var Buf: TVgaBuf);
var
  ER: TRect;
begin
  ER.Copy(R);
  ER.Intersect(Buf.Bounds);
  if ER.Empty then Exit;
  if not NoGraphics and (BackScreen <> nil) then begin
    if IndirectGraphics then begin
      if (P.X <> R.A.X) or (P.Y <> R.A.Y) then Exit; { not supported }
      gl_setcontext(ShadowScreen);
      gl_disableclipping;
      gl_copyboxfromcontext(BackScreen^, 
			    ER.A.X, ER.A.Y, ER.B.X - ER.A.X, ER.B.Y - ER.A.Y,
			    ER.A.X, ER.A.Y);
      UpdatePhysicalScreen(ER);
    end
    else begin
      gl_copyboxfromcontext(BackScreen^,
			    ER.A.X, ER.A.Y, ER.B.X - ER.A.X, ER.B.Y - ER.A.Y,
			    P.X, P.Y);
    end;
  end;
end;

procedure UpdatePhysicalScreen(var ER: TRect);
begin
  case PhysicalScreen^.ModeType of
    CONTEXT_PLANAR16: begin
      vga_copytoplanar16(PChar(ShadowScreen^.VBuf) + ER.A.Y * ShadowScreen^.ByteWidth + (ER.A.X and not 7),
			 ShadowScreen^.ByteWidth,
			 (ShadowScreen^.ByteWidth div 8) * ER.A.Y  + ER.A.X div 8,
			 ShadowScreen^.ByteWidth div 8,
			 ((ER.B.X - 1) or 7) - (ER.A.X and not 7) + 1,
			 ER.B.Y - ER.A.Y);
    end;
    CONTEXT_MODEX: begin
      vga_copytoplanar256(PChar(ShadowScreen^.VBuf) + ER.A.Y * ShadowScreen^.ByteWidth + (ER.A.X and not 3),
			  ShadowScreen^.ByteWidth,
			  (ShadowScreen^.ByteWidth div 4) * ER.A.Y  + ER.A.X div 4,
			  ShadowScreen^.ByteWidth div 4,
			  ((ER.B.X - 1) or 3) - (ER.A.X and not 3) + 1,
			  ER.B.Y - ER.A.Y);
    end;
  end;  
end; { UpdatePhysicalScreen }

function StoreScreen(x1, y1, x2, y2: Integer): PScreenBuf;
var
  s: LongInt;
  p: pointer;
  SaveOrigin: TPoint;

function NewScreenBuf(AMode: Word; AnInfo: LongInt): PScreenBuf;
 var
   p: PScreenBuf;
 Begin
   New(p);
   p^.Mode := AMode;
   p^.Size := s;
   p^.Rect.Assign(x1, y1, x2, y2);
   p^.Info := AnInfo;
   NewScreenBuf := p
 End;

Begin
{  if IndirectGraphics then begin StoreScreen := nil; exit end;}
  { General Images }
  s := 0;
  SaveOrigin := DrawOrigin;
  SetDrawOrigin(0, 0);
  p := CutImage(x1, y1, x2-1, y2-1);
  SetDrawOriginP(SaveOrigin);
  If p <> nil
    then StoreScreen := NewScreenBuf(2, LongInt(p))
  else StoreScreen := nil
End;

procedure FreeScreenBuf(Buf: PScreenBuf);
Begin
  If Buf <> nil then Begin
    case Buf^.Mode of
      2	: FreeImage(pointer(Buf^.Info));
    end;
    Dispose(Buf)
  End
End;

procedure DrawScreenBufAt(Buf: PScreenBuf; x3, y3: Integer);
var
  SaveOrigin: TPoint;
  R: TRect;
Begin
  If Buf <> nil then
    case Buf^.Mode of
      2	:
	  begin
	    R := Buf^.rect;
	    R.Intersect(ClipRect);
	    if not R.empty then begin
	      SaveOrigin := DrawOrigin;
	      SetDrawOrigin(0, 0);
	      PasteImage(x3, y3, pointer(Buf^.Info), NormalPut);
	      UpdatePhysicalScreen(R)
	    end;
	    SetDrawOriginP(SaveOrigin);
	  end
    end
End;

procedure DrawScreenBuf(Buf: PScreenBuf);
Begin
  If Buf <> nil then
    DrawScreenBufAt(Buf, Buf^.Rect.A.x, Buf^.Rect.A.y)
End;

function GetVgaMemCaps: Word;
begin
  if IndirectGraphics
    then GetVgaMemCaps := 0 { FIMXE: copy not quite ok yet }
    else GetVgaMemCaps := vmcCopy
end;

procedure GetTextMetrics(var Metrics: TTextMetric);
begin
  with Metrics do
  begin
    tmHeight := FontHeight;
    tmAscent := FontHeight;
    tmDescent := 0;
    tmInternalLeading := 0;
    tmExternalLeading := 0;
    tmAveCharWidth := FontWidth;
    tmMaxCharWidth := FontWidth;
    tmWeight := 700;
    tmItalic := 0;
    tmUnderlined := 0;
    tmStruckOut := 0;
    tmFirstChar := 0;
    tmLastChar := 255;
    tmDefaultChar := 32;
    tmBreakChar := 32;
    tmPitchAndFamily := 0;
    tmCharSet := 0;
    tmOverhang := 0;
    tmDigitizedAspectX := 100;
    tmDigitizedAspectY := 100
  end;
end;

procedure InitFonts;
var
  f: File;
  s: Integer;
type
  pp = ^pointer;
begin
  assign(f, {'/usr/lib/kbd/consolefonts/default8x16'} 'iso01.f14');
  {$i-}Reset(f, 1);{$i+}
  FontBuf := (pp(@gl_font8x8))^;
  FontWidth := 8;
  FontHeight := 8;
  if IOResult = 0 then begin
    s := filesize(f);
    GetMem(FontBuf, s);
    BlockRead(f, FontBuf^, s);
    FontHeight := s div 256;
    close(f);
  end;
end;

begin
  { Give up root permissions if we are root.
  }
  if geteuid = 0
    then vga_init;
  StdGrMode := vga_getdefaultmode;
  if StdGrMode = -1 then StdGrMode := gr800x600x256;
{$endif OS_Linux}

end.

