program HowToAccessFonts;

{$X+}

{ A little how-to program that shows how to access and output
  several types of fonts using MyFonts and the MyFonts TrueType link.

  Copr. 1995 Matthias Kppe.
}

uses
  Crt,
  Dos,
  Memory, 	{ TV memory management unit }
  Gr,		{ Graphics management unit }
  Bgi, 		{ A modified interface to the Graph unit }
  Misc,		{ Miscellaneous routines }
  MyFonts,	{ Font management unit }
  TTMyF;	{ MyFonts TrueType link }

var
  WinDir: string;

{ This procedure shows how to output a TrueType font.
}
procedure RegularTrueType;
var
  FontHandle: Integer;
begin
  { Create a font handle using LoadTTFont. This function does not
    support transformations. Both X and Y scaling are set to 70 pt.
  }
  FontHandle := DefFont(
    LoadTTFont(WinDir + 'system\timesbi.ttf', 70, 70));
  { Choose the defined font, and select a character spacing of 25 pixels,
    switch on text markers, and set white (15) text color and light red
    (12) selected-text color.
  }
  SetTextParams(FontHandle, 0{25}, 15 + 256 * 12, true);
  { Set text alignment using a Graph-compatible procedure.
    The set of valid vertical alignments has been extended to
    [TopText, LeadLine, CenterText, BaseLine, BottomText].
  }
  SetTextJustify(CenterText, BaseLine);
  { Output text using a Graph-compatible procedure.
  }
  outtextxy(320, 150, '~T~rue~T~ype');
  { Free the font handle.
  }
  FreeFont(FontHandle)
end;

{ This procedure shows how to output a rotated TrueType font using
  a transformation matrix created by MakeRotationMatrix.
}
procedure RotatedTrueType;
var
{ A font handle
}
  FontHandle: Integer;
{ Space for a transformation matrix
}
  Matrix: TTransMatrix;
begin
  { Create a font handle using LoadTTFontExt. This function supports
    specifying a transformation matrix. Both X and Y scales are set
    to 100 pt.
    You can define arbitrary sequences of transformations on the
    character glyph.
    * Identity fills Matrix with the unit matrix;
    * Italicize combines this matrix with a shearing transformation
      for synthesizing an italic font with an angle of 20 degrees;
    * Rotate combines this transformation with a counterclockwise
      rotation by 30 degrees.
    The last parameter decides how to draw rotated texts:
    * ftVerticalMoves allows vertical movement between the characters.
    * ftNoVerticalMoves aligns all characters to one hoizontal line.
  }
  FontHandle := DefFont(
    LoadTTFontExt(WinDir + 'system\arial.ttf', 100, 100,
      Rotate(+30, Italicize(+20, Identity(@Matrix))), ftNoVerticalMoves
    ));
  { Choose the defined font, and select a character spacing of 4 pixels,
    switch off text markers, and set white (15) text color.
  }
  SetTextParams(FontHandle, 4, 15, false);
  { Set text alignment using a Graph-compatible procedure.
    The set of valid vertical alignments has been extended to
    [TopText, LeadLine, CenterText, BaseLine, BottomText].
  }
  SetTextJustify(CenterText, BaseLine);
  { Output text using a Graph-compatible procedure.
  }
  outtextxy(320, 300, 'TrueType');
  { Free the font handle.
  }
  FreeFont(FontHandle)
end;

{ This procedure shows how to output bitmapped or scalable Windows fonts.
}
procedure WindowsFonts;
var
  SansSerif, Script: Integer;
begin
  { Create a font handle for a 24pt sans serif font.
  }
  SansSerif := DefFont(
    LoadWinFont(WinDir + 'system\sserife.fon',
      0,		{ don't choose by sequential number }
      24, 		{ choose pt size 24 }
      0, 		{ don't choose by pixel size }
      5, 		{ 5 pixels additional character spacing }
      0, 0, 		{ these are unimportant correction parameters }
      ftNormal		{ attribute to be synthesized }
    ));
  { Create a font handle for a 60-pixel emboldened and re-spaced
    script font.
  }
  Script := DefFont(
    LoadWinFont(WinDir + 'system\script.fon',
      0, 		{ don't choose by sequential number }
      0, 		{ don't choose by pt size }
      60, 		{ scale to a pixel height of 60 }
      0, 		{ 0 pixels additional character spacing }
      0, 0,
      ftBold		{ attribute to be synthesized }
    ));
  { Set text alignment
  }
  SetTextJustify(CenterText, BaseLine);
  { Select the sans serif font, using no additional character spacing,
    and a light blue color.
  }
  SetTextParams(SansSerif, 0, 9, false);
  { Draw some text.
  }
  OutTextXY(320, 420, 'Bitmapped Windows Font');
  { Select the script font, using no additional character spacing,
    light green / light yellow color, markers enabled
  }
  SetTextParams(Script, 0, 10 + 256 * 14, true);
  { Draw some text.
  }
  OutTextXY(320, 80, 'S~c~a~l~a~b~l~e~ W~i~n~d~o~w~s~ F~o~n~t');
  { Free the font handles.
  }
  FreeFont(SansSerif);
  FreeFont(Script)
end;

{ This procedure shows how to output a scalable BGI font,
  with a synthesized bold font attribute for a better look.
}
procedure BgiFont;
var
  FontHandle: Integer;
begin
  { Create a font handle for a 70-pixel emboldened gothic font.
  }
  FontHandle := DefFont(
    LoadBGIFileFont(PathToDriver + '\GOTH.CHR',
      70,               { choose a pixel height of 80 }
      0, 0, 0, 0, 	{ don't care about scaling and ratios }
      0, 		{ no character spacing }
      0, 0, 		{ unimportant correction parameters }
      ftBold));		{ synthesize a bold font version }
  { Select the font, using no additional character spacing, and
    brown color.
  }
  SetTextParams(FontHandle, 0, 6, false);
  { Set text alignment
  }
  SetTextJustify(CenterText, BaseLine);
  { Draw some text.
  }
  OutTextXY(320, 220, 'BGI Font');
  { Free the font handle.
  }
  FreeFont(FontHandle)
end;

{ This procedure shows how to output VGA BIOS and DOS CPI fonts.
}
procedure BIOSandCPIfonts;
var
  BIOS8, BIOS16, CPI14: Integer;
  egacpi: PathStr;
begin
  { Create some BIOS font handles.
  }
  BIOS8 := DefFont(LoadBIOSFont(8, 0, 0, 0));		{ 8-pixel font }
  BIOS16 := DefFont(LoadBIOSFont(16, 0, 0, 0));		{ 16-pixel font }
  { Read a CPI file and create a 14-pixel font handle from code-page 852.
  }
  egacpi := FSearch('ega.cpi', GetEnv('PATH'));
  CPI14 := DefFont(LoadCPIFont(egacpi, 852, 14, 0, 0, 0));
  { Select fonts and alignments and draw some text.
  }
  SetTextJustify(LeftText, BaseLine);
  SetTextParams(BIOS8, 0, 15, false);
  OutTextXY(0, 450, 'BIOS Font, 8 pixels');

  SetTextJustify(CenterText, BaseLine);
  SetTextParams(CPI14, 0, 15, false);
  OutTextXY(320, 470, 'CPI Font, 14 pixels: h   ב .');

  SetTextJustify(RightText, BaseLine);
  SetTextParams(BIOS16, 0, 15, false);
  OutTextXY(640, 450, 'BIOS Font, 16 pixels');
  { Free the font handles.
  }
  FreeFont(BIOS8);
  FreeFont(BIOS16);
  FreeFont(CPI14)
end;

begin
  { Initialization
  }
  setgrmode(gr640x480x16);
  initgraphics;
  WinDir := GetWinDir;

  { Higher buffer memory increases performance
  }
  MaxBufMem := 256 * 1024 div 16;
  initmyfonts;

  { How to use a regular TrueType font
  }
  RegularTrueType;
  readkey;

  { How to use a transformed TrueType font
  }
  RotatedTrueType;
  readkey;

  { How to use bitmapped or scalable Windows fonts
  }
  WindowsFonts;
  readkey;

  { How to use a scalable BGI font
  }
  BGIFont;
  readkey;

  { How to use BIOS and CPI fonts
  }
  BIOSandCPIfonts;
  readkey;

  { Shutdown
  }
  donemyfonts;
  closegraphics
end.
