// Symmetric-TSP solver
// Written by Matthias K"oppe <mkoeppe@csmd.cs.uni-magdeburg.de>
//
// Branch and bound, using a 1-tree relaxation with Lagrangean costs.
// Implemented branching rules: Smith and Thompson (1977), or 
// Volgenant and Jonker (1982). By change of `TOneNodePQ', you can
// switch between depth-first and breadth-first rule. For generation
// of tours (for upper bounds), a simple simulated-annealing algorithm
// is used.
//
// Usage:  1tree cont26.dat 100
// where `cont26.dat' is the data-file (which must be in the format of
// TSPLIB's NODE_COORD_SECTION) and `100' is the scale by which the
// coordinates are multiplied.

#include "branch.h"
#include "anneal.h" // used as upper-bounding device
#include <set>

int Scale;

const int Inf = 1000000;

typedef IndexMatrix<int> TIntMatrix;
typedef Matrix<int> TRawIntMatrix;

// 1-tree relaxation

class Edge : public pair<int, int> {
public:
  Edge() {}
  Edge(int i, int j) : pair<int, int>(i, j) {}
  int Other(int vertex) const {
    if (vertex == first) return second;
    else if (vertex == second) return first;
    else return -1;
  }
  friend ostream &operator<<(ostream &s, const Edge &E) {
    return s << E.first << "--" << E.second;
  }
  friend int operator==(const Edge &A, const Edge &B) {
    return A.first==B.first && A.second==B.second
      || A.first==B.second && A.second==B.first;
  }
};

class EdgeVector : public vector<Edge> {
public:
  int Degree(int Vertex) const {
    const_iterator i;
    int deg = 0;
    for (i = begin(); i!=end(); ++i)
      if (Vertex==(*i).first || Vertex==(*i).second) deg++;
    return deg;
  }
  pair<int, int> UnbranchedDegree(int Vertex, const TIntMatrix &M) const {
    const_iterator i;
    pair<int, int> deg = pair<int, int>(0, 0);
    for (i = begin(); i!=end(); ++i)
      if ((Vertex==(*i).first || Vertex==(*i).second)) {
	deg.first++;
	if ((M((*i).Other(Vertex), Vertex) != 0)) deg.second++;
      }
    return deg;
  }
  friend ostream &operator<<(ostream &s, const EdgeVector &EV) {
    copy(EV.begin(), EV.end(), ostream_iterator<Edge>(s, " "));
    return s;
  }
};

template <class T>
EdgeVector MinimalSpanningTree(IndexMatrix<T> &cost,
			       int First, int Last) //return Tree
  // Ermittelt einen minimalen aufspannenden Baum f"ur den
  // Teilgraphen mit Knoten First bis Last. "
{
  // Prim-Dijkstra
  EdgeVector Tree;
  typedef set<int, less<int> > IntSet;
  IntSet Coming;
  vector<int> Connect;
  int i;
  Connect.push_back(0);
  for (i = First + 1; i<=Last; i++) {
    Connect.push_back(First); // really First
    Coming.insert(i);
  }
  while (!Coming.empty()) {
    T MinCost = 2 * Inf * Scale;
    IntSet::iterator v, MinV;
    for (v = Coming.begin(); v != Coming.end(); ++v) {
      T Cost = cost(*v, Connect[*v - First]);
      if (Cost < MinCost) {
	MinCost = Cost;
	MinV = v;
      }
    }
    int MinVV = *MinV;
    Coming.erase(MinV);
    Tree.push_back(Edge(MinVV, Connect[MinVV - First]));
    for (v = Coming.begin(); v != Coming.end(); ++v) {
      if (cost(*v, Connect[*v - First]) > cost(*v, MinVV))
	Connect[*v - First] = MinVV;
    }
  }
  return Tree;
}

template <class T>
EdgeVector MinimalOneTree(IndexMatrix<T> &cost) //return OneTree
// Ermittelt einen minimalen 1-Tree.
{
  EdgeVector OneTree = MinimalSpanningTree(cost, 1, cost.Size);
  // F"uge zwei Kanten, von 0 beginnend, hinzu. "
  T MinCost = 3 * Inf * Scale;
  int MinI, MinJ;
  int i, j;
  for (i = 1; i<cost.Size; i++)
    for (j = i+1; j<=cost.Size; j++) {
      T Cost = cost(0, i) + cost(0, j);
      if (Cost < MinCost) {
	MinCost = Cost;
	MinI = i, MinJ = j;
      }
    }
  OneTree.push_back(Edge(0, MinI));
  OneTree.push_back(Edge(0, MinJ));
  return OneTree;
}

template <class T>
T Cost(const Edge &E, const IndexMatrix<T> &cost)
{
  return cost(E.first, E.second);
}

template <class T>
T Cost(const EdgeVector &EV, const IndexMatrix<T> &cost)
{
  T c = 0;
  EdgeVector::const_iterator i;
  for (i = EV.begin(); i!=EV.end(); ++i)
    c+=Cost(*i, cost);
  return c;
}

int VertexWithDegreeNotTwo(const EdgeVector &EV, int Begin, int End)
{
  int i;
  for (i = Begin; i<=End; ++i)
    if (EV.Degree(i) != 2) return i;
  return -1;
}

int VertexWithUnbranchedDegreeNotTwo
(const EdgeVector &EV, int Begin, int End, const TIntMatrix &M)
{
  int i;
  for (i = Begin; i<=End; ++i) {
    pair<int, int> deg = EV.UnbranchedDegree(i, M);
    if (deg.first != 2 && deg.second > 0) return i;
  }
  return -1;
}

int VertexWithUnbranchedDegreeGreaterThanTwo
(const EdgeVector &EV, int Begin, int End, const TIntMatrix &M)
{
  int i;
  for (i = Begin; i<=End; ++i) {
    pair<int, int> deg = EV.UnbranchedDegree(i, M);
    if (deg.first > 2 && deg.second > 0) return i;
  }
  return -1;
}

int IsTour(const EdgeVector /* 1-tree */ &EV, int Begin, int End) 
{
  // It suffices to check the degree.
  return VertexWithDegreeNotTwo(EV, Begin, End) == -1;
}

// GNU call it `times'
template <class T>
struct multiplies : binary_function<T, T, T> {
    T operator()(const T& x, const T& y) const { return x * y; }
};

template <class T>
class NumVector : public vector<T> {
public:
  NumVector(int siz) : vector<T>(siz, 0) {}
  friend NumVector operator* (T scalar, const NumVector &vec) 
  { 
    NumVector res(vec.size());
    transform(vec.begin(), vec.end(),
	      res.begin(),
	      bind2nd(multiplies<T>(), scalar));
    return res;
  }
  NumVector &operator+= (const NumVector &vec)
  {
    transform(begin(), end(),
	      vec.begin(),
	      begin(),
	      plus<T>());
    return *this;
  }
  friend ostream &operator<< (ostream &s, const NumVector &vec)
  {
    copy(vec.begin(), vec.end(), ostream_iterator<T>(s, " "));
    return s;
  }
    
};

typedef NumVector<double> TVector;
typedef IndexMatrix<double> TCostMatrix;

template <class T>
IndexMatrix<T> UpdateCost(const IndexMatrix<T> &cost,
			  const NumVector<T> &lambda,
			  T &ForceCost) 
{
  IndexMatrix<T> M(cost);
  vector<T>::const_iterator i, j;
  int k, l;
  T MinCost = *min_element(lambda.begin(), lambda.end());
//   cout << "UpdateCost: MinCost " << MinCost
//        << " MaxCost " << *max_element(lambda.begin(), lambda.end())
//        << endl;

  T IncludeCost;
  ForceCost = 0;
  if (MinCost > 0) IncludeCost = 0;
  else IncludeCost = 3 * MinCost - 10;

  for (k = 0, i = lambda.begin(); i!=lambda.end(); ++i, k++)
    for (l = 0, j = lambda.begin(); j!=lambda.end(); ++j, l++) 
      if (M(k, l)==0) { // this is a hack that ensures that the edges
			// marked to be included (cost 0) are
			// included, independently of lambda.
	M(k, l) = IncludeCost;
	ForceCost -= IncludeCost - ((*i)+(*j));
      }
      else M(k, l) += (*i)+(*j);
  
  ForceCost /= 2;
  return M;
}

class Incident : public unary_function<Edge, int> {
  int Vertex;
public:
  Incident(int vertex) : Vertex(vertex) {}
  int operator()(const Edge &edge) {
    return edge.first == Vertex || edge.second == Vertex;
  }
};

Tour TourOfEdgeVector(EdgeVector EV)
{
  Tour tour(EV.size());
  EdgeVector::iterator i = EV.begin();
  int vertex = (*i).first;
  int j = 0;
  do {
    vertex = (*i).Other(vertex);
    tour[j++] = vertex;
    EV.erase(i);
    i = find_if(EV.begin(), EV.end(), Incident(vertex));
  } while (i!=EV.end());
  return tour;
}

int MaxBound;
Tour BestTour(0);
bool BranchTour = false;

void FoundTour(const EdgeVector &EV, int Cost)
{
  
 //  cout << "SolveDual: Found tour "
//        << EV
//        << "with a length of "
//        << Cost << endl;
  
  BranchTour = true;
  Tour tour = TourOfEdgeVector(EV);
  int Len = length(tour);
  //cout << "Actual length is " << Len << endl;
  if (Len < MaxBound) {
    MaxBound = Len;
    BestTour = tour;
    cout << '\n' << BestTour
	 << " Length " << ((double) length(BestTour))/Scale<<endl;     
  }
  
}

pair<EdgeVector, int>
SolveDual(const TIntMatrix &cost,
	  int UpperBound, int FixCost,
	  double StartAlpha = 2.0,
	  double AlphaFactor = 0.9,
	  double Epsilon =  0.001)
{
  // Subgradient optimization method for the Lagrangean dual of the
  // TSP, as described in Lawler et al. (1985), p373.
  
  // Numerical calculation is performed in floating-point.
  typedef pair<EdgeVector, int> Result;
  // Start with a Lagrangean vector of zero;
  // no cost updating necessary
  TVector lambda(cost.Size+1);
  TCostMatrix NormalCost(cost, 4711); 
  TCostMatrix UpdatedCost(NormalCost);
  double alpha = StartAlpha;
  // Well the procedure is said to converge to the optimum, but the
  // easiest way to get a bound is to fetch the maximum.
  double BestBound = 0;
  EdgeVector BestOneTree;
  double step;
  double ForceCost = 0;
  
  do {
    EdgeVector OneTree = MinimalOneTree(UpdatedCost);
    double OneTreeCost = Cost(OneTree, UpdatedCost)
      - 2 * accumulate(lambda.begin(), lambda.end(), 0.0)
      + ForceCost
      + FixCost;
    if (OneTreeCost > BestBound) {
      BestBound = OneTreeCost;
      BestOneTree = OneTree;
    }
//     cout << "At lambda = " << lambda << " we have an optimal 1-tree "
// 	 << OneTree << " with a length of " << OneTreeCost << endl;
    if (IsTour(OneTree, 0, cost.Size)) {
//       cout << "NormalCost says "
// 	   << Cost(OneTree, NormalCost) << "plus fix " <<  FixCost << endl;

      FoundTour(OneTree, (int) floor(OneTreeCost));
      return Result(OneTree, MAX_INT); // (int) floor(OneTreeCost));
    }
    if (OneTreeCost >= UpperBound)
      return Result(OneTree, (int) floor(OneTreeCost));
    
    TVector Subgradient(cost.Size+1);
    int i;
    for (i = 0; i<=cost.Size; i++) 
      Subgradient[i] = OneTree.Degree(i) - 2;
//     cout << "Subgradient " << Subgradient << endl;
    step = alpha * (UpperBound + 10 * Scale - OneTreeCost)   //& 100 is a
						      //hack for test 
      / inner_product(Subgradient.begin(), Subgradient.end(),
		      Subgradient.begin(), 0.0);
//     cout << "Step " << step << endl;

    lambda += step * Subgradient;
    alpha *= AlphaFactor;
    UpdatedCost = UpdateCost(NormalCost, lambda, ForceCost);

  } while (step > Epsilon);
  return Result(BestOneTree, (int) floor(BestBound));
}

int AnzZimmer;

TRawIntMatrix *MakeDistMatrix() 
{
  TRawIntMatrix *Matrix = new TRawIntMatrix(AnzZimmer+1, AnzZimmer+1);
  int i, j;
  for (i = 0; i<=AnzZimmer; i++)
    for (j = 0; j<=AnzZimmer; j++)
      if (i==j) (*Matrix)[i][j] = Inf * Scale;
      else (*Matrix)[i][j]
	     = (int) rint(DoubleDist(cities[i], cities[j]) * Scale); 
  return Matrix;
}

class TOneNode {
public:
  EdgeVector OneTree;
  int Bound;
  int Depth;
  TIntMatrix *Matrix;
  int FixCost;

  bool ShouldDelete;
  
  TOneNode() : Matrix(0), Depth(-1), Bound(Inf * Scale) {};
  TOneNode(TIntMatrix *matrix,
	   int fixcost, int depth, int Upper) :
    Matrix(matrix), FixCost(fixcost), Depth(depth)
  {
    pair<EdgeVector, int> One = SolveDual(*Matrix, Upper, FixCost);
    OneTree = One.first;
    Bound = One.second;
    cout << Bound << ":" << MaxBound << " " << flush;
    // Debugging: Check if included edges have been taken.
    int i, j;
    for (i=0; i<=Matrix->Size; i++)
      for (j=0; j<=Matrix->Size; j++)
	if ((*Matrix)(i, j) == 0) {
	  //cout << "!";
	  if (find(OneTree.begin(), OneTree.end(), Edge(i, j)) == OneTree.end())
	    cout << "Huh? at " << i << ":" << j << endl;
	}
  }
  TOneNode(const TOneNode &N) :
    OneTree(N.OneTree), Bound(N.Bound), Depth(N.Depth),
    Matrix(N.Matrix ? new TIntMatrix(*N.Matrix) : 0), FixCost(N.FixCost) {}
  TOneNode &operator=(const TOneNode &N) {
    OneTree = N.OneTree;
    Bound = N.Bound;
    Depth = N.Depth;
    if (Matrix) delete Matrix;
    if (N.Matrix) Matrix = new TIntMatrix(*N.Matrix);
    else Matrix = 0;
    FixCost = N.FixCost;
    return *this;
  }      
  ~TOneNode() {
    if (Matrix) delete Matrix;
  }
			
  friend operator<(const TOneNode &A, const TOneNode &B)
  {
    if (A.Depth < B.Depth) return 1;
    if (A.Depth > B.Depth) return 0;
    return A.Bound > B.Bound;
  } 
  friend operator&&(const TOneNode &A, const TOneNode &B)
  {
    return A.Bound > B.Bound;
  }    
};

template <class T>
class CountPair {
  friend class CountRef<T>;
  int Count;
  T Content;
public:
  CountPair() : Count(0) {}
  CountPair(T &content) : Count(1), Content(content) {
    content.ShouldDelete = false;
  }
  void Increase() {Count++;}
  void Decrease() {if (--Count==0) delete this;}
};

template <class T>
class CountRef {
  CountPair<T> *Value;
public:
  CountRef() : Value(0) {}
  CountRef(T &content) : Value(new CountPair<T>(content)) {}
  CountRef(const CountRef &C) : Value(C.Value) {if (Value) Value->Increase();}
  ~CountRef() {if (Value) Value->Decrease();}
  CountRef &operator=(const CountRef &C) {
    if (Value) Value->Decrease();
    Value = C.Value;
    if (Value) Value->Increase();
    return *this;
  }
  operator T&() {return Value->Content;}
  //  operator const T&() const {return Value->Content;}
  //  operator void*() const {if (Value) return (void *)Value->Content; else return 0;}
};


class TOneNodePQ : 
  public priority_queue<vector<CountRef<TOneNode> >, logical_and /*less*/ <TOneNode> > {
public:
    TOneNodePQ() {}
};

TOneNodePQ NodePQ;

void SplitNode(TOneNode &Node) // Smith and Thompson (1977)
{
  if (Node.Bound >= MaxBound) return;
  // Choose vertex with degree not 2.
  cout << "Split: bound " << Node.Bound << " " << flush;
  int NotTwo = VertexWithUnbranchedDegreeNotTwo
    (Node.OneTree, 0, Node.OneTree.size()-1, *Node.Matrix); 
  if (NotTwo == -1) {
    // In diesem 1-tree ist das Nicht-Tour-Sein festgezurrt.
    // Wir k"onnen diesen Knoten einfach vergessen. "
//     int i;
//     for (i = 0; i<Node.OneTree.size(); i++) {
//       pair<int, int> deg = Node.OneTree.UnbranchedDegree(i, *Node.Matrix);
//       cout << i << " deg(" << deg.first << ", " << deg.second << endl;
//     }
    return;
  }
  // Find maximum-cost edge (NotTwo, MaxVertex)
  EdgeVector::const_iterator i;
  int MaxCost = -1;
  int MaxVertex;
  for (i = Node.OneTree.begin(); i!=Node.OneTree.end(); ++i) {
    int j = (*i).Other(NotTwo); // returns -1 if not incident
    if (j>=0) {
      int Cost = (*Node.Matrix)(NotTwo, j);
      if (Cost > MaxCost) {
	MaxCost = Cost;
	MaxVertex = j;
      }
    }
  }
//   cout << "Pivot edge is " << Edge(MaxVertex, NotTwo) << " MaxCost " <<
//        MaxCost << endl;
  // Create successor nodes
  {
    TIntMatrix *ExcludeMatrix = new TIntMatrix(*Node.Matrix);
    (*ExcludeMatrix)(NotTwo, MaxVertex) = Inf * Scale;
    (*ExcludeMatrix)(MaxVertex, NotTwo) = Inf * Scale;
    TOneNode Exclude(ExcludeMatrix,
		     Node.FixCost, Node.Depth+1, MaxBound);
    if (Exclude.Bound < MaxBound) NodePQ.push(Exclude);
  }
  {
    TIntMatrix *IncludeMatrix = Node.Matrix;
    Node.Matrix = 0;
    int FixCost = Node.FixCost + (*IncludeMatrix)(NotTwo, MaxVertex);
    //    cout << "Include: FixCost " << FixCost;
    (*IncludeMatrix)(NotTwo, MaxVertex) = 0;
    (*IncludeMatrix)(MaxVertex, NotTwo) = 0;
    TOneNode Include(IncludeMatrix,
		     FixCost, Node.Depth+1, MaxBound);
    if (Include.Bound < MaxBound) NodePQ.push(Include);
  }
  cout << endl;
}

bool CheckFixedDegree(TIntMatrix &M, int Node)
{
  // if included edges make up degree 2,
  // exclude other edges explicitly.
  int i;
  int deg = 0;
  for (i = 0; i<=M.Size; i++) 
    if (M(i, Node) == 0) deg++;
  if (deg > 2) {
    return false;
  }
  else if (deg == 2) {
    for (i = 0; i<=M.Size; i++) 
      if (M(i, Node) != 0) {
	M(i, Node) = Inf * Scale;
	M(Node, i) = Inf * Scale;
      }
  }
  return true;
}

void SplitNodeVJ(TOneNode &Node) // Volgenant and Jonker (1982),
				 // slightly modified
{
  if (Node.Bound >= MaxBound) return;
  // Choose vertex with degree not 2.
  cout << "Split: bound " << Node.Bound << " " << flush;
  int NotTwo = VertexWithUnbranchedDegreeGreaterThanTwo
    (Node.OneTree, 0, Node.OneTree.size()-1, *Node.Matrix); 
  if (NotTwo == -1) {
    // In diesem 1-tree ist das Nicht-Tour-Sein festgezurrt.
    // Wir k"onnen diesen Knoten einfach vergessen. "
//     int i;
//     for (i = 0; i<Node.OneTree.size(); i++) {
//       pair<int, int> deg = Node.OneTree.UnbranchedDegree(i, *Node.Matrix);
//       cout << i << " deg(" << deg.first << ", " << deg.second << endl;
//     }
    return;
  }
  // Find two free edges incident with NotTwo
  EdgeVector::const_iterator i;
  int j = -1;
  int k = -1;
  bool IncludedIncident = false;

  int MaxCost = -1;
  int MaxVertex;

  for (i = Node.OneTree.begin(); i!=Node.OneTree.end(); ++i) {
    j = (*i).Other(NotTwo);
    if (j>=0) {
      int Cost = (*Node.Matrix)(NotTwo, j);
      if (Cost > MaxCost) {
	MaxCost = Cost;
	MaxVertex = j;
      }
      // if ((*Node.Matrix)(NotTwo, j) != 0) break;
      if (!Cost) IncludedIncident = true;
    }
  }
  j = MaxVertex;

  for (i = Node.OneTree.begin(); i!=Node.OneTree.end(); ++i) {
    k = (*i).Other(NotTwo);
    if (k>=0 && k!=j) {
      if ((*Node.Matrix)(NotTwo, k) != 0) break;
      IncludedIncident = true;
    }
  }

  if (!IncludedIncident) {
    int l;
    for (; i!=Node.OneTree.end(); ++i) {
      l = (*i).Other(NotTwo);
      if (l>=0) {
	if ((*Node.Matrix)(NotTwo, l) == 0) {
	  IncludedIncident = true;
	  break;
	}
      }
    }
  }
  if (k<0) { // did not find two free edges
    cerr << "This must be special-cased." << flush;

    

  }
  else {
    
    int t = j; j=k; k=t;
    
    // included-incident logic is buggy.

    // Create successor nodes
    //if (!IncludedIncident) { // k1
    { TIntMatrix *K1Matrix = new TIntMatrix(*Node.Matrix);
      int FixCost = Node.FixCost
	+ (*K1Matrix)(NotTwo, j)
	+ (*K1Matrix)(NotTwo, k);
      int l;
      for (l = 0; l<=(*Node.Matrix).Size; l++) {
	if (l!=j && l!=k) {
	  (*K1Matrix)(NotTwo, l) = Inf * Scale;
	  (*K1Matrix)(l, NotTwo) = Inf * Scale;
	}
	else {
	  (*K1Matrix)(NotTwo, l) = 0;
	  (*K1Matrix)(l, NotTwo) = 0;
	}
      }
      if (CheckFixedDegree(*K1Matrix, NotTwo)
	  && CheckFixedDegree(*K1Matrix, l)) {
	TOneNode K1(K1Matrix,
		    FixCost, Node.Depth+1, MaxBound);
	if (K1.Bound < MaxBound) NodePQ.push(K1);
      }
      else {
	cout << "Dropped k1 " << flush;
	delete K1Matrix;
      }
    }
    { // k2
      TIntMatrix *K2Matrix = new TIntMatrix(*Node.Matrix);
      int FixCost = Node.FixCost + (*K2Matrix)(NotTwo, j);
      (*K2Matrix)(NotTwo, k) = Inf * Scale;
      (*K2Matrix)(k, NotTwo) = Inf * Scale;
      (*K2Matrix)(NotTwo, j) = 0;
      (*K2Matrix)(j, NotTwo) = 0;
      if (CheckFixedDegree(*K2Matrix, NotTwo)
	  && CheckFixedDegree(*K2Matrix, j)) {
	TOneNode K2(K2Matrix,
		    FixCost, Node.Depth+1, MaxBound);
	if (K2.Bound < MaxBound) NodePQ.push(K2);
      }
      else {
	cout << "Dropped k2 " << flush;
	delete K2Matrix;
      }
    }
    { // k3
      TIntMatrix *K3Matrix = Node.Matrix;
      Node.Matrix = 0;
      (*K3Matrix)(NotTwo, j) = Inf * Scale;
      (*K3Matrix)(j, NotTwo) = Inf * Scale;
      TOneNode K3(K3Matrix,
		  Node.FixCost, Node.Depth+1, MaxBound);
      if (K3.Bound < MaxBound) NodePQ.push(K3);
    }
    cout << endl;
  }
}


// upper-bounding device
template <class Neighbourhood, class Accept>
void MyGeneralIteration(Neighbourhood &neighbourhood, Accept &accept,
		      Tour &tour, int MaxIter)
{
  int i;
  int tourlength = length(tour);
  for (i = 0; i<MaxIter; i++) {
    Tour next = tour;
    neighbourhood(next);
    int nextlength = length(next);
    if (accept(nextlength, tourlength)) {
      tourlength = nextlength;
      tour = next;
      if (tourlength < MaxBound) {
	MaxBound = tourlength;
	BestTour = tour;
      }
    }
  }
}

int main(int argc, char **argv)
{
  ifstream data(argv[1]);
  if (argc>=3) Scale = atoi(argv[2]);
  else Scale = 10;
    
  ReadCities(data);
  distances = CalcDistances(cities, Scale);
  AnzZimmer = cities.size() - 1;
  TIntMatrix *Matrix = new TIntMatrix(AnzZimmer, MakeDistMatrix());
  //   EdgeVector OneTree = MinimalOneTree(*Matrix);
  //   cout << OneTree << " Length " << Cost(OneTree, *Matrix) << endl;
  //  cout << "Bound " << SolveDual(*Matrix, 5000) << endl;

  //MaxBound = Inf * Scale; // 1000 * Scale;

  // simple tour for upper bound
//   MaxBound = (*Matrix)(0, AnzZimmer);
//   int i;
//   for (i = 0; i<AnzZimmer; i++) 
//     MaxBound += (*Matrix)(i, i+1);

//   MaxBound = 1000 * Scale;

  Tour tour(cities.size());

  const int temp = 30;
  const double factor = 0.26;
  const iter = 1000000;
  const iterstep = 10000;

  SimulatedAnnealing anneal(temp, iter * factor);

  //SimulatedAnnealingNoReject(anneal, tour2, iter);
  //GeneralIteration(ChangeTwoNH(), anneal, tour2, iter);

  MaxBound = Inf * Scale;
  MyGeneralIteration(Lin2OptNH(), anneal, tour, iterstep);
//   MaxBound = length(tour);
//   BestTour = tour;

  //
  NodePQ.push(TOneNode());
  NodePQ.push(TOneNode(Matrix, 0, 0, MaxBound));
  while (NodePQ.size() > 1) {
    cout << "PQ size " << NodePQ.size() << " " << flush;
    CountRef<TOneNode> Node = NodePQ.top();
    NodePQ.pop();
    SplitNodeVJ(Node);

    if (!BranchTour) { 
      // If we haven't found a tour by branching yet,
      // do some annealing steps to improve the upper bound.
      MyGeneralIteration(Lin2OptNH(), anneal, tour, iterstep);
//       int tourlength = length(tour);
//       if (tourlength < MaxBound) {
// 	MaxBound = tourlength;
// 	BestTour = tour;
//       }
    }
  }
  cout << '\n' << BestTour
       << " Length " << ((double) length(BestTour))/Scale;     

  char resname[200];
  strcpy(resname, argv[1]);
  strcat(resname, ".1t.res");
  ofstream res(resname, ios::app|ios::out);
  res << BestTour
      << " Length " << ((double) length(BestTour))/Scale << endl;     
}

