// ESSO: Extensible Shell with Scheme Orientation
// Written by Matthias Koeppe <mkoeppe@csmd.cs.uni-magdeburg.de>

#if defined(_LowLevel)
#include "bpipe.h"
#include "mboxpipe.h"
extern "C" {
#include "library/libfile.h"
#include "library/syslib.h"
#include "library/iolib.h"
#include "library/error.h"
#include "system/mailbox.h"
#include "system/loader.h"
#include "system/process.h"
}
#else
extern "C" {
#include <signal.h>
#include <errno.h>
#include <unistd.h>
}
#include <stdio.h>
#include <fstream.h>
#include "stpipe.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <sys/time.h>
#include <time.h>
#endif
#include "lisp.h"
#include <stack>
#include <vector>

TLispRef *_GlobalFrame;
TLispRef *_LispNull, *_LispTrue, *_LispFalse;
string *SyntaxAListName;

void AssertArgCount(TLispRef ArgList, int Count)
{
  int i;
  for (i = 0; i < Count; i++) {
    if (!ArgList.IsPair()) THROW(XArgCount(i, Count));
    ArgList = ArgList.CDr();
  }
  if (!ArgList.IsNull()) THROW(XArgCount(Count + 1, Count));
}

// #define AssertArgCount0(A) AssertArgCount(A, 0)
// #define AssertArgCount1(A) AssertArgCount(A, 1)
// #define AssertArgCount2(A) AssertArgCount(A, 2)
// #define AssertArgCount3(A) AssertArgCount(A, 3)
// #define AssertArgCount4(A) AssertArgCount(A, 4)
// #define AssertArgCount5(A) AssertArgCount(A, 5)

#define AssertArgCount0(A) AssertNull(A)
#define AssertArgCount1(A) AssertNull(A.Cdr())
#define AssertArgCount2(A) AssertNull(A.Cddr())
#define AssertArgCount3(A) AssertNull(A.Cdddr())
#define AssertArgCount4(A) AssertNull(A.Cddddr())
#define AssertArgCount5(A) AssertNull(A.Cdddddr())

TLispRef LispEq(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  if (ArgList.CAr().Type() == ArgList.CADr().Type()
      && ArgList.CAr().IsEq(ArgList.CADr()))
    return LispTrue;
  return LispFalse;
}

TLispRef LispType(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return new TLispSymbol(ArgList.CAr().TypeName());
}

TLispRef LispCar(TLispRef ArgList) 
{
  AssertArgCount1(ArgList);
  return ArgList.CAr().Car();
}

TLispRef LispCdr(TLispRef ArgList) 
{
  AssertArgCount1(ArgList);
  return ArgList.CAr().Cdr();
}

TLispRef LispCons(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  return new TLispPair(ArgList.CAr(), ArgList.CADr());
}

TLispRef LispList(TLispRef ArgList)
{
  //& actually make newly allocated list
  return ArgList;
}

TLispRef LispSetCar(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  ArgList.CAr().Car() = ArgList.CADr();
  return ArgList.CAr();
}

TLispRef LispSetCdr(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  ArgList.CAr().Cdr() = ArgList.CADr();
  return ArgList.CAr();
}

TLispRef LispMakePromise(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return new TLispPromise(ArgList.CAr());
}

#ifdef _LowLevel
const int task_force_stack_size = 32768;
const int task_force_stack_hack_stack_size = 512;
char task_force_stack_hack_stack[task_force_stack_hack_stack_size];
// Task force: Since no one knows that we exist, we have to get rid of
// our large dynamic stack ourselves. For this purpose, we switch to a
// static mini-stack, hoping that no other task force does the same
// thing at the same time. 
extern "C" void task_force_stack_hack(void *stack_start)
{
  asm("movl %%eax, %%esp\n"
      "pushl %%edx\n"
      "call sys_free\n"
      "call sys_exit_process"
      : : "a" (task_force_stack_hack_stack
	       + task_force_stack_hack_stack_size),
      "d" (stack_start)); 
}
// Arguments for task_force_process
static TLispRef *task_force_ref;
static char *stack_start;

extern "C" void task_force_process(void)
{
  void *my_stack_start = stack_start;
  TLispRef *my_task_force_ref = task_force_ref;
  TRY {
    TRY {
      my_task_force_ref->Type(); // will force the value.
    } 
#if defined(OWNEXCEPTIONS)
      else { delete Exception; Exception = 0; }
#else
      catch(...) {};
#endif
//       kprintf("uuu");
      delete my_task_force_ref; // We don't need it any more.
    }
#if defined(OWNEXCEPTIONS)
      else { delete Exception; Exception = 0; }
#else
      catch(...) {};
#endif
//       kprintf("ooo");
      UnregisterThreadWithTries();
//       kprintf("ppp");
  // will do sys_exit_process():
  task_force_stack_hack(my_stack_start);
}

#endif

TLispRef LispTaskForce(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  // Ensure that the Car is not discarded in the parent's thread of
  // execution. Moreover, this is the only way to pass an `argument'
  // to the new process.
  task_force_ref = new TLispRef(ArgList.CAr());
  // create a new process
  stack_start = (char *) sys_malloc(task_force_stack_size);

  RegisterThreadWithTries(stack_start, stack_start + task_force_stack_size);

  int pid = sys_create_process(task_force_process,
			       stack_start, task_force_stack_size); 
  return LispTrue;
#else
  // Ensure that the Car is not discarded in the parent's thread of
  // execution.
  TLispRef *Ref = new TLispRef(ArgList.CAr());
  // create a new process
  // Note we must use a shared-memory operator new.
  // Otherwise all this makes no sense.
  pid_t pid = fork(); 
  // I tried to use clone(SIGCLD, 0), but it is not present in the
  // current libc. The direct system call did not work properly
  // either. 
  if (!pid) { // this thread is the child
    //    raise(SIGSTOP); // debugging
    TRY {
      TRY {
	Ref->Type(); // will force the value.
      } 
#if defined(OWNEXCEPTIONS)
      else { delete Exception; Exception = 0; }
#else
      catch(...) {};
#endif
      delete Ref; // We don't need the Car any more.
    }
#if defined(OWNEXCEPTIONS)
      else { delete Exception; Exception = 0; }
#else
      catch(...) {};
#endif
    _exit(0); // terminate, but don't call destructors
  }
  return LispTrue;
#endif
}

TLispRef LispFrameBindings(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  TLispFrame::TLispMap *Map = &(AssertFrame(ArgList.CAr())->GetMap());
  TLispFrame::TLispMap::const_iterator i = Map->begin();
  TLispRefOS Result(LispNull);
  TLispRef *ResultPtr = &Result;
  while (i != Map->end()) {
    *ResultPtr = new TLispPair(new TLispSymbol((*i).first), LispNull);
    ResultPtr = &ResultPtr->Cdr();
    ++i;
  }
  return Result;
}

TLispRef LispFrameParent(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return AssertFrame(ArgList.CAr())->GetParent();
}

TLispRef LispSymbol2String(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return new TLispString(ArgList.CAr().SymbolName());
}

TLispRef LispString2Symbol(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return new TLispSymbol(AssertString(ArgList.CAr())->Value());
}

TLispRef LispApply(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  return ArgList.CAr().Apply(ArgList.CADr());
}

TLispRef LispPlus(TLispRef ArgList)
{
  int Res = 0;
  TLispRefOS Cur(ArgList);
  while (Cur.IsPair()) {
    Res += AssertNumber(Cur.CAr())->Value();
    Cur = Cur.CDr();
  }
  AssertNull(Cur);
  return NEW TLispNumber(Res);
}

TLispRef LispTimes(TLispRef ArgList)
{
  int Res = 1;
  TLispRefOS Cur(ArgList);
  while (Cur.IsPair()) {
    Res *= AssertNumber(Cur.CAr())->Value();
    Cur = Cur.CDr();
  }
  AssertNull(Cur);
  return NEW TLispNumber(Res);
}

TLispRef LispMinus(TLispRef ArgList)
{
  if (ArgList.Cdr().IsNull())
    return NEW TLispNumber(- AssertNumber(ArgList.CAr())->Value());
  AssertNull(ArgList.CdDr());
  return NEW TLispNumber(AssertNumber(ArgList.CAr())->Value()
			 - AssertNumber(ArgList.CADr())->Value());
}

TLispRef LispQuotient(TLispRef ArgList)
{
  AssertNull(ArgList.Cddr());
  if (AssertNumber(ArgList.CADr())->Value()==0) return LispFalse;
  return NEW TLispNumber(AssertNumber(ArgList.CAr())->Value()
			 / AssertNumber(ArgList.CADr())->Value());
}

template <class T>
TLispRef LispNumberMonotonous(TLispRef ArgList,
			      const T &Compare)
{
  if (ArgList.IsNull()) return LispTrue;
  int Res = AssertNumber(ArgList.Car())->Value();
  TLispRefOS Cur(ArgList.CDr());
  while (Cur.IsPair()) {
    int NewRes;
    if (!(Compare(Res, (NewRes = AssertNumber(Cur.CAr())->Value()))))
      return LispFalse;
    Res = NewRes;
    Cur = Cur.CDr();
  }
  AssertNull(Cur);
  return LispTrue;
}

TLispRef LispNumberEqual(TLispRef ArgList)
{
  return LispNumberMonotonous(ArgList, equal_to<int>());
}

TLispRef LispNumberLess(TLispRef ArgList)
{
  return LispNumberMonotonous(ArgList, less<int>());
}

TLispRef LispNumberLessEqual(TLispRef ArgList)
{
  return LispNumberMonotonous(ArgList, less_equal<int>());
}

TLispRef LispNumberGreater(TLispRef ArgList)
{
  return LispNumberMonotonous(ArgList, greater<int>());
}

TLispRef LispNumberGreaterEqual(TLispRef ArgList)
{
  return LispNumberMonotonous(ArgList, greater_equal<int>());
}

TLispRef LispChar2Integer(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return NEW TLispNumber(AssertChar(ArgList.CAr())->Value());
}

TLispRef LispInteger2Char(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return NEW TLispChar(AssertNumber(ArgList.CAr())->Value());
}

TLispRef LispStringSet(TLispRef ArgList)
{
  AssertArgCount3(ArgList);
  string *s = &AssertString(ArgList.CAr())->Value();
  int n = AssertNumber(ArgList.CADr())->Value();
  if (n >= 0 && n < s->length()) {
    (*s)[n] = AssertChar(ArgList.CADDr())->Value();
    return LispNull;
  }
  else THROW(XBadStringIndex());
}

TLispRef LispStringRef(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  int n = AssertNumber(ArgList.CADr())->Value();
  string *s = &AssertString(ArgList.CAr())->Value();
  if (n >= 0 && n < s->length())
  return NEW TLispChar((*s)[n]);
  else THROW(XBadStringIndex());
}

TLispRef LispStringLength(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return NEW TLispNumber
    (AssertString(ArgList.CAr())->Value().length());     
}    

TLispRef LispMakeString(TLispRef ArgList)
{
  char c;
  if (ArgList.Cdr().IsNull()) c = ' ';
  else c = AssertChar(ArgList.CaDr())->Value();
  int n = AssertNumber(ArgList.CAr())->Value();
  string s;
  int i;
  for (i = 1; i<=n; i++) s+=c;
  return new TLispString(s);
}

TLispRef LispStringAppend(TLispRef ArgList)
{
  string s;
  TLispRefOS Cur(ArgList);
  while (Cur.IsPair()) {
    s += AssertString(Cur.CAr())->Value();
    Cur = Cur.CDr();
  }
  AssertNull(Cur);
  return new TLispString(s);
}

TLispRef LispSubstring(TLispRef ArgList)
{
  AssertArgCount3(ArgList);
  int start = AssertNumber(ArgList.CADr())->Value();
  int end = AssertNumber(ArgList.CADDr())->Value();
  if (end > start)
    return new TLispString(AssertString(ArgList.CAr())
			   ->Value().substr(start, end-start));
  return new TLispString("");
}

TLispRef LispStringMember(TLispRef ArgList)
{
  // (string-member string from-pos char) returns the first pos from
  // from-pos where the char occurs, or #f.
  AssertArgCount3(ArgList);
  string *s = &AssertString(ArgList.CAr())->Value();
  int frompos = AssertNumber(ArgList.CADr())->Value();
  char c = AssertChar(ArgList.CADDr())->Value();
  int i;
  for (i = frompos; i < s->length(); i++)
    if ((*s)[i] == c) return NEW TLispNumber(i);
  return LispFalse;
}

TLispRef LispStringEq(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  return (AssertString(ArgList.CAr())->Value()
	  == AssertString(ArgList.CADr())->Value())
    ? LispTrue : LispFalse;
}

TLispRef LispRead(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  //& This is a hack... We must change lisp_read to
  // work with TLispPort instead of pipes.
  TPipeBase<char> *p = AssertInputPort(ArgList.CAr())->Pipe;
  if (p) return lisp_read(*p);
  return NEW TLispEof();
}

TLispRef LispWrite(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  //& This is a hack... We must change lisp_write to
  // work with TLispPort instead of pipes.
  TPipeBase<char> *p = AssertOutputPort(ArgList.CADr())->Pipe;
  if (p) {
    lisp_write(*p, ArgList.CAr());
    return LispTrue;
  }
  else return LispFalse;
}

TLispRef LispReadChar(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  char c = (AssertInputPort(ArgList.CAr())->ReadChar());
  if (c=='\004') return NEW TLispEof();
  return NEW TLispChar(c);
}

TLispRef LispWriteChar(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  char c = (AssertChar(ArgList.CAr()))->Value();
  return AssertOutputPort(ArgList.CADr())->WriteChar(c) ? LispTrue : LispFalse;
}

TLispRef LispReadLine(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  if (AssertInputPort(ArgList.CAr())->Eof())
    return NEW TLispEof();
  return new TLispString((CastInputPort(ArgList.CAr())->ReadLine()));
}

TLispRef LispWriteLine(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  return (AssertOutputPort(ArgList.CADr())->
    WriteLine(AssertString(ArgList.CAr())->Value())) ? LispTrue : LispFalse;
}

TLispRef LispFileExists(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  direntry info;
  char name[250];
  strcpy(name, "a:");
  strcat(name, AssertString(ArgList.CAr())->Value().c_str());
  return (stat(name, &info) >= 0) ? LispTrue : LispFalse; 
#else
  ifstream is(AssertString(ArgList.CAr())->Value().c_str());
  if (is) return LispTrue;
  else return LispFalse;  
#endif
}

TLispRef LispFileDirectory(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  direntry info;
  char name[250];
  strcpy(name, "a:");
  strcat(name, AssertString(ArgList.CAr())->Value().c_str());
  if (stat(name, &info)>=0)
    if (info.attrib & 0x10) return LispTrue;
  return LispFalse;
#else
  struct stat info;
  if (stat(AssertString(ArgList.CAr())->Value().c_str(), &info) == -1) return LispFalse;
  if (info.st_mode & S_IFDIR) return LispTrue;
  return LispFalse;
#endif
}

TLispRef LispFileExecutable(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  // We have no `executable' file mode in the low-level environment.
  // Executables are recognized via file extension.
  return LispFalse;
#else
  struct stat info;
  if (stat(AssertString(ArgList.CAr())->Value().c_str(), &info) == -1) return LispFalse;
  if (info.st_mode & S_IXUSR) return LispTrue;
  return LispFalse;
#endif
}

TLispRef LispOpenInputFile(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  int des, size;
  DOSfile fd;
  char *data;
  direntry info;
  char name[250];
  strcpy(name, "a:");
  strcat(name, AssertString(ArgList.CAr())->Value().c_str());
  des = open(&fd, name, 2);
  //& insert file mode 
  if (stat(name, &info)<0)
    return LispFalse;
  data = new char[info.size + 1];
  int bytes;
  if ((bytes = read(&fd, data, info.size))<0) {
    close(&fd);
    delete[] data;
    return LispFalse;
  }
  close(&fd);
  data[bytes] = 0;
  TBlockReadPipe *brp = new TBlockReadPipe(data);
  return new TLispInputPort(brp);
#else
  ifstream *is = new
    ifstream(AssertString(ArgList.CAr())->Value().c_str());
  if (*is) {
    TIstreamPipe *isp = new TIstreamPipe(is);
    return new TLispInputPort(isp);
  }
  delete is;
  return LispFalse;
#endif
}

TLispRef LispOpenOutputFile(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  //& File writes have not yet been defined in `shell_fs.h'
  return LispFalse;
#else
  ofstream *os = new
    ofstream(AssertString(ArgList.CAr())->Value().c_str());
  if (*os) {
    TOstreamPipe *osp = new TOstreamPipe(os);
    return new TLispOutputPort(osp);
  }
  delete os;
  return LispFalse;
#endif
}

TLispRef LispCloseInputPort(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  AssertInputPort(ArgList.CAr())->Close();
  return LispTrue;
}

TLispRef LispCloseOutputPort(TLispRef ArgList)
{
//   kprintf("cop");
  AssertArgCount1(ArgList);
  AssertOutputPort(ArgList.CAr())->Close();
//   kprintf("poc");
  return LispTrue;
}

TLispRef LispMakePortReadPromise(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
  return new TLispPortReadPromise(ArgList.CAr());
}

TLispRef LispHandlePortReadPromise(TLispRef ArgList)
{
  AssertArgCount2(ArgList);
  // If the first arg is not a port-read-promise, return false. Otherwise
  // assume that the second arg is an output port; copy everything
  // from the port-read-promise to the output port, and return true.
  TLispRef First = ArgList.CAr();
  TLispPromiseBase *P;
  TLispInputPort *I;
  do { // force all other promises
    P = First.AsPromise();
    if (P) {
      I = P->InputPort();
      if (!I) First = P->Force();
    }
  } while (P && !I);
  if (P && I) { 
    // We have a port-read-promise; copy everything.
    TLispOutputPort *O = AssertOutputPort(ArgList.CADr());
    while (!I->Eof()) {
      /* FIXME: Read larger chunks than chars */
      char C = I->ReadChar();
      if (C != '\004') {
	if (!O->WriteChar(C)) {
//  	  kprintf("::");
	  return LispTrue;
	}
	// This happens when the process which we write to has been
	// killed (see `mboxpipe.h').
      }
    }
    return LispTrue;
  }
  return LispFalse;
}

TLispRef LispRunExecutable(TLispRef ArgList)
{
  // return a pair of exec-input and exec-output
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  char name[250];
  strcpy(name, "a:");
  strcat(name, AssertString(ArgList.CAr())->Value().c_str());
  prog_info *pi = new prog_info;
  if (loader(name, pi) < 0) return LispFalse;

  TMailBox *inmbox = new TMailBox(100);  /* proc input, ESSO output */
  TMailBox *outmbox = new TMailBox(100); /* proc output, ESSO input */ 
  
  char *stack = new char[4096];
  int pid = sys_create_io_process(pi->entry, stack, 4096, *inmbox, *outmbox);

  // The ombox will not be deleted when we get rid of the
  // omboxpipe. This will be done by the imboxpipe; see `mboxpipe.h'.
  TOMailBoxPipe *omboxpipe = new TOMailBoxPipe(*inmbox /* STAR */);
  TIMailBoxExecPipe *imboxpipe
    = new TIMailBoxExecPipe(outmbox /* NO STAR */, omboxpipe, pi, pid, stack);
  return new TLispPair(new TLispOutputPort(omboxpipe, false),
		       new TLispInputPort(imboxpipe));
#else
  /* FIXME: This FIFO stuff is messy, but I don't know what to do else */
  static int fifocount = 0;
  char command[500];
  strcpy(command, AssertString(ArgList.CAr())->Value().c_str());
  fifocount++;
  char fifoname[20];
  sprintf(fifoname, "/tmp/essofifo.#%d", fifocount);
  mkfifo(fifoname, 0666);
  strcat(command, " < ");
  strcat(command, fifoname);
  FILE *f = popen(command, "r");
  if (!f) return LispFalse;
  ofstream *os = new ofstream(fifoname);
  return new TLispPair(new TLispOutputPort(new TOstreamPipe(os)),
		       new TLispInputPort(new TIstreampPipe/*note p*/(f)));

#endif
}

TLispRef LispLoadDirectory(TLispRef ArgList)
{
  AssertArgCount1(ArgList);
#ifdef _LowLevel
  TLispRef Result = LispNull;
  DOSfile fd;
  direntry dir;
  char name[250];
  strcpy(name, "a:");
  strcat(name, AssertString(ArgList.CAr())->Value().c_str());
  if (opendir(&fd, name)<0) return Result;
  TLispRef *ResultPtr = &Result;

  while (readdir(&fd, &dir, 1)) {
    char *nm;
    char dosname[12];
    if (dir.longname[0]) {
      nm = dir.longname;
    }
    else {
      strncpy(dosname, dir.name, 8);
      char *p = dosname;
      p[8] = 0;
      while (*p > ' ') p++;
      if (dir.ext[0] > ' ') {
	*p++ = '.';
	p[3] = 0;
	strncpy(p, dir.ext, 3);
	while (*p > ' ') p++;
      }
      *p = 0;
      /* convert to lowercase */
      for (int i = 0; i<12; i++)
	if (dosname[i] >= 'A' && dosname[i] <= 'Z')
	  dosname[i] += ('a' - 'A');
      nm = dosname;
    }
    *ResultPtr = new TLispPair(new TLispString(nm), LispNull);
    ResultPtr = &ResultPtr->CDr();
  }
  return Result;
#else
  DIR *dir;
  struct dirent *line;
  dir = opendir(AssertString(ArgList.CAr())->Value().c_str()); 
  if (!dir) return LispFalse;
  else {
    TLispRefOS Result(LispNull);
    TLispRef *ResultPtr = &Result;
    do {
      line = readdir(dir);
      if (line) {
	*ResultPtr = new TLispPair(new TLispString(line->d_name), LispNull);
	ResultPtr = &ResultPtr->CDr();
      }
    } while (line);
    closedir(dir);
    return Result;
  }
#endif
}

TLispRef LispSystemTime(TLispRef ArgList)
{
#if defined(_LowLevel)
  return NEW TLispNumber(0);
#else
  struct timeval tv;
  struct timezone tz;
  gettimeofday(&tv, &tz);
  return NEW TLispNumber((time(0) % 1800) * 1000000 + tv.tv_usec);
#endif
}

TLispRef MakeGlobalFrame()
{
  TLispFrame *Frame = new TLispFrame(LispNull);
  Frame->Define("nil", LispNull);
  Frame->Define("the-global-frame", Frame);
  Frame->Define("syntax-alist", LispNull);
  Frame->Define("eq?", new TLispPrimitive(LispEq));
  Frame->Define("type", new TLispPrimitive(LispType));
  Frame->Define("car", new TLispPrimitive(LispCar));
  Frame->Define("cdr", new TLispPrimitive(LispCdr));
  Frame->Define("cons", new TLispPrimitive(LispCons));
  Frame->Define("list", new TLispPrimitive(LispList)); 
  Frame->Define("set-car!", new TLispPrimitive(LispSetCar));
  Frame->Define("set-cdr!", new TLispPrimitive(LispSetCdr));
  Frame->Define("make-promise", new TLispPrimitive(LispMakePromise));
  Frame->Define("task-force", new TLispPrimitive(LispTaskForce));
  Frame->Define("frame-bindings", new TLispPrimitive(LispFrameBindings));
  Frame->Define("frame-parent", new TLispPrimitive(LispFrameParent));
  Frame->Define("symbol->string",
		new TLispPrimitive(LispSymbol2String));
  Frame->Define("string->symbol",
		new TLispPrimitive(LispString2Symbol));
  Frame->Define("apply", new TLispPrimitive(LispApply));
  Frame->Define("+", new TLispPrimitive(LispPlus));
  Frame->Define("*", new TLispPrimitive(LispTimes));
  Frame->Define("-", new TLispPrimitive(LispMinus));
  Frame->Define("quotient", new TLispPrimitive(LispQuotient));
  Frame->Define("=", new TLispPrimitive(LispNumberEqual));
  Frame->Define("<", new TLispPrimitive(LispNumberLess));
  Frame->Define(">", new TLispPrimitive(LispNumberGreater));
  Frame->Define("<=", new TLispPrimitive(LispNumberLessEqual));
  Frame->Define(">=", new TLispPrimitive(LispNumberGreaterEqual));

  Frame->Define("char->integer", new TLispPrimitive(LispChar2Integer));
  Frame->Define("integer->char", new TLispPrimitive(LispInteger2Char));
  Frame->Define("string-set!", new TLispPrimitive(LispStringSet));
  Frame->Define("string-ref", new TLispPrimitive(LispStringRef));
  Frame->Define("string-length", new TLispPrimitive(LispStringLength));
  Frame->Define("make-string", new TLispPrimitive(LispMakeString));
  Frame->Define("string-append", new TLispPrimitive(LispStringAppend));
  Frame->Define("string-member", new TLispPrimitive(LispStringMember));
  Frame->Define("substring", new TLispPrimitive(LispSubstring));
  Frame->Define("string=?", new TLispPrimitive(LispStringEq));

  Frame->Define("read", new TLispPrimitive(LispRead));
  Frame->Define("read-char", new TLispPrimitive(LispReadChar));
  Frame->Define("read-line", new TLispPrimitive(LispReadLine));
  Frame->Define("write", new TLispPrimitive(LispWrite));
  Frame->Define("write-char", new TLispPrimitive(LispWriteChar));
  Frame->Define("write-line", new TLispPrimitive(LispWriteLine));
  Frame->Define("file-exists?", new TLispPrimitive(LispFileExists));
  Frame->Define("file-directory?", new TLispPrimitive(LispFileDirectory));
  Frame->Define("file-executable?", new TLispPrimitive(LispFileExecutable));
  Frame->Define("open-input-file", new TLispPrimitive(LispOpenInputFile));
  Frame->Define("open-output-file", new TLispPrimitive(LispOpenOutputFile));
  Frame->Define("close-input-port", new TLispPrimitive(LispCloseInputPort));
  Frame->Define("close-output-port", new TLispPrimitive(LispCloseOutputPort));
  Frame->Define("handle-port-read-promise",
		new TLispPrimitive(LispHandlePortReadPromise));
  Frame->Define("make-port-read-promise",
		new TLispPrimitive(LispMakePortReadPromise));
  Frame->Define("run-executable",
		new TLispPrimitive(LispRunExecutable));
  Frame->Define("load-directory", new TLispPrimitive(LispLoadDirectory));
  Frame->Define("system-time", new TLispPrimitive(LispSystemTime));
  return Frame;
}

#if !defined(HEAVY)
void AssertType(TLispRef &ref, TLispBase::TLispType type)
{
  if (ref.Type() != type) 
    ThrowInvalidType(ref.Type(), type);
}
#endif

void TLispObject::DeleteThis()
{
  delete this;
}

TLispRef TLispObject::Apply(TLispRef ArgList)
{
  THROW(XApply(Type()));
}

void ThrowInvalidType(TLispBase::TLispType objecttype, TLispBase::TLispType type)
{
  THROW(XInvalidType(objecttype, type));
}

#if defined(OWNEXCEPTIONS)

// void TLispRefOS::Register()
// {
//   Next = First;
//   First = this;
// }

void TLispRefOS::Unregister()
{
  TLispRefOS **P = &First;
  while (*P && *P!=this) P=&(*P)->Next;
  if (*P) { // should always be true
    *P = Next;
  }
}

void TLispRefOS::KillAllBetween(void *Lower, void *Higher)
{
  TLispRefOS *P = First;
  while (P) {
    TLispRefOS *Next = P->Next;
    if (P > Lower && P < Higher) delete P; // will change this list
    P = Next;
  }
}
#endif

void TLispRef::DoForce() //throw()
{
  do {
    *this = ((TLispPromiseBase *)ObjectPtr())->Force();
  } while (object->Type() == TLispBase::Promise);
}

TLispRef TLispPromise::Force()
{
  if (!Ready) {
    Value = Expression.Apply(LispNull);
    Expression = LispNull;
    Ready = true;
  }
  return Value;
}

TLispRef TLispPortReadPromise::Force()
{
  if (!Ready) {
    // This is coined after the old lazy-load-text
    if (CastInputPort(Port)->Eof()) Value = LispNull;
    else Value = new TLispPair
	   (new TLispString((CastInputPort(Port)->ReadLine())),
	    new TLispPortReadPromise(Port));
    Ready = true;
    Port = LispFalse;
  }
  return Value;
}

TLispRef *TLispFrame::find(const TLispName &Key) 
{
  TLispFrame *Frame = this;
  do {
    TLispMap::iterator I = Frame->Map.find(Key);
    if (I != Frame->Map.end()) return &(*I).second;
    if (Frame->Parent.IsNull()) Frame = 0;
    else Frame = CastFrame(Frame->Parent);
  } while (Frame);
  return 0;
}

void TLispFrame::ThrowXNameNotFound(const TLispName &Key)
{
  THROW(XNameNotFound(Key));
}

void TLispFrame::Define(const TLispName &name, TLispRef ref)
{
  pair<TLispMap::iterator,
    bool> result = Map.insert(TLispMap::value_type(name, ref));
  if (!result.second) {
    // already defined in this frame; replace value.
    (*result.first).second = ref;
  }
}

void TLispLambda::ApplyAux(TLispRef ArgList,
			       /*returns*/ TLispRef &RetExpression,
			       /*returns*/ TLispRef &RetFrame)
{
  // Bind formals to actuals.
  TLispRefOS ArgFrame(new TLispFrame(Frame));
  
  TLispRefOS CurFormal(Formals);
  TLispRefOS CurActual(ArgList);
  while (CurFormal.IsPair()) {
    CastFrame(ArgFrame)->Define(CurFormal.CAr().SymbolName(),
				  CurActual.Car());
    CurFormal = CurFormal.CDr();
    CurActual = CurActual.CDr();
  }
  if (CurFormal.IsSymbol()) {
    // rest-list formal argument
    CastFrame(ArgFrame)->Define(CurFormal.SymbolName(),
				CurActual);
  }
  // set returns
  RetExpression = Body;
  RetFrame = ArgFrame;
}

TLispRef TLispLambda::Apply(TLispRef ArgList)
{
  // This should not be used (not properly tail-recursive)
  TLispRefOS RetExpression, ArgFrame;
  ApplyAux(ArgList, RetExpression, ArgFrame);
  // Evaluate body
  return lisp_eval(RetExpression, ArgFrame);  
}

char TLispInputPort::ReadChar()
{
  if (Pipe && !eof) {
    while (!CurrentPos || !*CurrentPos) {
      if (CurrentBlock) delete CurrentBlock;
      CurrentBlock = CurrentPos = Pipe->Get();
    }
    if (*CurrentPos == '\004') eof = true;
    return *CurrentPos++;
  }
  eof = true;
  return '\004';
}

string TLispInputPort::ReadLine()
{
  string s;
  if (Pipe) {
    while (!CurrentPos || (*CurrentPos != '\n' && *CurrentPos != '\004')) {
      if (!CurrentPos || !*CurrentPos) {
	if (CurrentBlock) delete CurrentBlock;
	CurrentBlock = CurrentPos = Pipe->Get();
      }
      if (*CurrentPos && *CurrentPos != '\n' && *CurrentPos != '\004') {
	s += *CurrentPos++;
      }
    }
    if (*CurrentPos == '\004') eof = true;
    if (*CurrentPos == '\n') CurrentPos++;
  }
  return s;
}

bool TLispOutputPort::WriteChar(char c)
{
  if (Pipe && !Pipe->Closed()) {
    char buf[2];
    buf[0] = c;
    buf[1] = 0;
    Pipe->Put(strdup(buf));
    return true;
  }
  return false;
}

bool TLispOutputPort::WriteLine(const string &s)
{
  if (Pipe && !Pipe->Closed()) {
    Pipe->Put(strdup(s.c_str()));
    Pipe->Put(strdup("\n"));
    return true;
  }
  return false;
}

#if defined(OWNEXCEPTIONS)
#define CATCHALL					\
  else {						\
  outputpipe.Put(strdup(Exception->Message().c_str()));	\
  outputpipe.Put(strdup("\n"));				\
  delete Exception;					\
  Exception = 0;					\
}

#else
#define CATCHALL						\
    catch (XNameNotFound &X) {					\
      outputpipe.Put(strdup(X.Message().c_str()));		\
      outputpipe.Put(strdup("\n"));				\
    }								\
    catch (XBadExternalSymbol &X) {				\
      outputpipe.Put(strdup(X.Message().c_str()));		\
      outputpipe.Put(strdup("\n"));				\
    }								\
    catch (XInvalidType &X) {					\
      outputpipe.Put(strdup(X.Message().c_str()));		\
      outputpipe.Put(strdup("\n"));				\
    }								\
    catch (XApply &X) {						\
      outputpipe.Put(strdup(X.Message().c_str()));		\
      outputpipe.Put(strdup("\n"));				\
    }								\
    catch (XBadStringIndex &X) {				\
      outputpipe.Put(strdup(X.Message().c_str()));		\
      outputpipe.Put(strdup("\n"));				\
    }								\
    catch (XArgCount &X) {					\
      outputpipe.Put(strdup(X.Message().c_str()));		\
      outputpipe.Put(strdup("\n"));				\
    }								\
    catch (...) {						\
      outputpipe.Put(strdup("Unexpected exception.\n"));	\
    }
#endif

void read_eval_write_loop(TTextPipe &inputpipe,
			  TTextPipe &commandpipe,
			  TTextPipe &outputpipe, 
			  TLispRef Frame)
{
  TLispRef result, expr, Prompt;
  string EssoRunning = "esso-running";
  TRY {
    AssertFrame(Frame)->Define(EssoRunning, LispTrue);
    CastFrame(Frame)->Define("the-current-input-port", new TLispInputPort(&inputpipe));
    CastFrame(Frame)->Define("the-current-output-port", new TLispOutputPort(&outputpipe));
    do {
//       kprintf("$");
      TRY {
	Prompt = lisp_eval(new TLispPair(new TLispSymbol("esso-input-prompt"),
					 LispNull) /*InputPromptExpr*/, Frame);
	outputpipe.Put(strdup(AssertString(Prompt)->Value().c_str()));
      } CATCHALL;
//       kprintf("&");
      TRY {
	expr = lisp_read(commandpipe);
// 	kprintf("/");
	result = lisp_eval(expr, Frame);
// 	kprintf("++");
	TRY {
	  Prompt = lisp_eval(new TLispPair(new TLispSymbol("esso-output-prompt"),
					    LispNull) /*OutputPromptExpr*/, Frame);
// 	  kprintf("--");
	  outputpipe.Put(strdup(AssertString(Prompt)->Value().c_str()));
	} CATCHALL;
	TRY {
	  lisp_write(outputpipe, result);
	  outputpipe.Put(strdup("\n"));
	} CATCHALL;
      } CATCHALL;
      result = LispNull;
      expr = LispNull;
    } while ((*CastFrame(Frame))(EssoRunning).IsTrue());
  } CATCHALL;
}

bool isws(char c) 
{
  return (c == '\t' || c == ' ' || c == '\n' || c == '\r');
}

void skipws(char *&s) 
{
  if (s) while (isws(*s)) s++;
}

void skipwsread(char *&s, char *&Buf, TTextPipe &pipe)
{
  do {
    skipws(s);
    if (!s || !*s) {
      if (Buf) delete[] Buf;
      s = Buf = pipe.Get();
    }
  } while (!s || !*s || *s == '\t' || *s == ' ' || *s == '\n' || *s == '\r');
}

class ListState {
public:
  TLispPair *CurrentList;
  TLispPair *FillMyCdr;
  bool ListStart;
  bool ListEnd;
  bool AutoClose;
  ListState::ListState() : CurrentList(0), FillMyCdr(0),
    ListStart(false), ListEnd(false), AutoClose(false) {}
};

TLispRef lisp_read(TTextPipe &pipe)
{
  ListState State;
  stack<vector<ListState> > StateStack;
  char *Buf = pipe.Get();
  char *P = Buf;
  bool EofRead = false;
  do {
    TLispRef Ref;
    bool HaveObj = false;
    if (!(State.AutoClose && State.ListEnd)) skipwsread(P, Buf, pipe);
    if ((State.AutoClose && State.ListEnd) || EofRead || *P == ')') {
      if (!((State.AutoClose && State.ListEnd) || EofRead)) P++;
      if (State.ListStart) {
	Ref = LispNull;
	HaveObj = true;
	State.ListStart = false;
      }
      else {
	Ref = State.CurrentList;
	HaveObj = true;
      }
      if (!StateStack.empty()) {
	State = StateStack.top();
	StateStack.pop();
      }
      else {
	State.FillMyCdr = 0;
	State.CurrentList = 0;
      }
    }
    else if (*P == '(') {
      P++;
      if (State.FillMyCdr || State.ListStart) {
	StateStack.push(State);
      }
      State.FillMyCdr = 0;
      State.CurrentList = 0;
      State.ListStart = true;
      State.ListEnd = false;
      State.AutoClose = false;
    }
    else if (*P == '.' && (isws(*(P+1)) || !*(P+1))) {
      P++;
      State.ListEnd = true;
    }
    else if (*P == '\'') {
      // quote
      P++;
      if (State.FillMyCdr || State.ListStart) {
	StateStack.push(State);
      }
      State.CurrentList = new TLispPair(new TLispSymbol("quote"),
					LispNull);
      State.FillMyCdr = State.CurrentList;
      State.ListStart = false;
      State.AutoClose = true;
    }
    else if (*P == '`') {
      // quasiquote
      P++;
      if (State.FillMyCdr || State.ListStart) {
	StateStack.push(State);
      }
      State.CurrentList = new TLispPair(new TLispSymbol("quasiquote"),
					LispNull);
      State.FillMyCdr = State.CurrentList;
      State.ListStart = false;
      State.AutoClose = true;
    }
    else if (*P == ',') {
      // unquote
      P++;
      if (State.FillMyCdr || State.ListStart) {
	StateStack.push(State);
      }
      State.CurrentList
	= new TLispPair(new TLispSymbol(*P == '@'
					? P++, "unquote-splicing"
					: "unquote"), 
			LispNull);
      State.FillMyCdr = State.CurrentList;
      State.ListStart = false;
      State.AutoClose = true;
    }
    else if (*P == '\004') { /* EOF */
      Ref = NEW TLispEof();
      HaveObj = true;
      EofRead = true;
    }
    else if (*P == '\003') { /* CANCEL */ 
      // Get rid of the whole stack of open lists
      if (State.CurrentList) delete State.CurrentList;
      while (!StateStack.empty()) {
	State = StateStack.top();
	StateStack.pop();
	if (State.CurrentList) delete State.CurrentList;
      }
      // Return False
      return LispFalse;
    }      
    else if (*P == '"') {
      // read a string
      string S;
      P++; // consume quote
      do {
	while (*P != '"' && *P) {
	  if (*P == '\\') {
	    P++;
	    if (*P == 'n') S+='\n';
	    else S+=*P;
	    P++;
	  }
	  else S += *P++;
	}
	if (!*P) {
	  delete[] Buf;
	  P = Buf = pipe.Get();
	}
      } while (*P != '"');
      P++; // consume quote
      Ref = new TLispString(S);
      HaveObj = true;
    }
    else if (*P == '#') {
      P++;
      if (*P == 't') {
	P++; 
	Ref = NEW TLispBoolean(true);
	HaveObj = true;
      } 
      else if (*P == 'f') {
	P++;
	Ref = NEW TLispBoolean(false);
	HaveObj = true;
      }
      else if (*P == '\\') {
	P++;
	string S;
	do {
	  S += *P++;
	} while (*P && *P != ' ' && *P != '\t' && *P != '\n' && *P != '\r'
		 && *P != ')' && *P != '(');
	if (S.length() == 1) Ref = NEW TLispChar(S[0]), HaveObj = true;
	else if (S=="space") Ref = NEW TLispChar(' '), HaveObj = true;
	else if (S=="newline") Ref = NEW TLispChar('\n'), HaveObj = true;
      }
    }
    else if (*P >= '0' && *P <= '9'
	     || *P=='-' && *(P+1) >= '0' && *(P+1) <= '9') {
      // read a number
      int number = 0;
      bool neg = *P == '-';
      if (neg) P++;
      do {
	number = number * 10 + (*P++ - '0');
      } while (*P >= '0' && *P <= '9');
      Ref = NEW TLispNumber(neg ? -number : number);
      HaveObj = true;
    }
    else {
      // read a symbol 
      string S;
      do {
	S += *P++;
      } while (*P && *P != ' ' && *P != '\t' && *P != '\n' && *P != '\r'
	       && *P != ')' && *P != '(');
      Ref = new TLispSymbol(S);
      HaveObj = true;
    }
    if (HaveObj) {
      if (State.ListStart) {
	State.CurrentList = State.FillMyCdr = new TLispPair(Ref, LispNull);
	State.ListStart = false;
      } 
      else if (State.FillMyCdr) {
	if (State.ListEnd) {
	  State.FillMyCdr->Cdr() = Ref;
	  State.ListEnd = false;
	}
	else {
	  if (State.AutoClose) State.ListEnd = true;
	  TLispPair *cdr = new TLispPair(Ref, LispNull);
	  State.FillMyCdr->Cdr() = TLispRef(cdr);
	  State.FillMyCdr = cdr;
	}
      } 
      else return Ref;
    }
  } while (1);
}

void lisp_write(TTextPipe &pipe, TLispRef Ref)
{
  // Copy port-read-promise
  TLispPromiseBase *P = Ref.AsPromise();
  if (P) {
    TLispInputPort *I = P->InputPort();
    if (I) {
      while (!I->Eof()) {
	char C[2];
	/* FIXME: Read larger chunks than chars */
	C[0] = I->ReadChar();
	C[1] = 0;
	if (C[0] != '\004')
	  pipe.Put(strdup(C));
      }
      I->Close();
      return;
    }
  }
  // Normal output
  switch (Ref.Type()) {
  case TLispBase::Null: 
    pipe.Put(strdup("()"));
    break;
  case TLispBase::Number:
    {
      int Num = CastNumber(Ref)->Value();
      if (Num == 0) pipe.Put(strdup("0"));
      else {
	bool Neg = (Num < 0);
	if (Neg) Num = -Num;
	char Ascii[12];
	Ascii[11] = 0;
	char *P = Ascii+11;
	while (Num) {
	  *--P = Num%10 + '0';
	  Num /= 10;
	}
	if (Neg) *--P = '-';
	pipe.Put(strdup(P));
      }
      break;
    }
  case TLispBase::Pair: 
    pipe.Put(strdup("("));
    lisp_write(pipe, Ref.CAr());
    while (Ref.CDr().IsPair()) {
      pipe.Put(strdup(" "));
      Ref = Ref.CDr();
      lisp_write(pipe, Ref.CAr());
    }
    if (!Ref.CDr().IsNull()) {
      pipe.Put(strdup(" . "));
      lisp_write(pipe, Ref.CDr());
    }
    pipe.Put(strdup(")")); 
    break;
  case TLispBase::String:
    pipe.Put(strdup("\""));
    pipe.Put(strdup(CastString(Ref)->Value().c_str()));
    pipe.Put(strdup("\""));
    break;
  case TLispBase::Symbol:
    pipe.Put(strdup(CastSymbol(Ref)->Name().c_str()));
    break;
  case TLispBase::Boolean:
    pipe.Put(strdup(Ref.IsTrue() ? "#t" : "#f"));
    break;
  case TLispBase::Char:
    {
      char c[4];
      c[2] = CastChar(Ref)->Value();
      switch (c[0]) {
      case ' ':
	pipe.Put(strdup("#\\space"));
	break;
      case '\n':
	pipe.Put(strdup("#\\newline"));
	break;
      default: 
	c[0] = '#'; c[1] = '\\'; c[3] = 0;
	pipe.Put(strdup(c));
      }
      break;
    }
  case TLispBase::Lambda:
  case TLispBase::Frame: 
  case TLispBase::Primitive:
  case TLispBase::Promise:
    pipe.Put(strdup(("[a " + Ref.TypeName() + "]").c_str()));
    break;
  case TLispBase::Eof:
  case TLispBase::InputPort: 
  case TLispBase::OutputPort:
  case TLispBase::Executable:
    pipe.Put(strdup(("[an " + Ref.TypeName() + "]").c_str()));
    break;
  }
}

TLispRef lisp_load(TTextPipe &Pipe, TLispRef Frame)
{
  TLispRefOS lastresult, result, expr;
  do {
    expr = lisp_read(Pipe);
    lastresult = result;
    result = lisp_eval(expr, Frame);
  } while (result.Type() != TLispBase::Eof);
  return lastresult;
}

void ThrowBadExternalSymbol(const string &S)
{
  THROW(XBadExternalSymbol(S));
}

TLispRef lisp_eval(TLispRef Ref, TLispRef Frame)
{
  while (1) {
    switch (Ref.Type()) {
    case TLispBase::Symbol:
      {
	// look up symbol's value
	TLispRef *Result = CastFrame(Frame)->find(Ref.SymbolName());
	if (Result) return *Result;
	// do external look-up
	// (define (eval-external-symbol symbol-name-string) ...)
	// return (list value) or '().
	TLispRef *EvalExt =
	  CastFrame(Frame)->find("eval-external-symbol");
	if (!EvalExt) THROW(XNameNotFound(Ref.SymbolName())); 
	else {
	  TLispRefOS ResList(EvalExt->Apply(new TLispPair(Ref,
							  LispNull)));
	  if (ResList.IsPair()) {
	    AssertNull(ResList.CDr());
	    return ResList.CAr();
	  }
	  ThrowBadExternalSymbol(Ref.SymbolName());
	}
      }
    case TLispBase::Pair:
      {
	if (Ref.CAr().IsSymbol()) {
	  string *Name = &Ref.CAr().SymbolName();
	  // check for special forms
	  if (*Name == "define") {
	    if (Ref.CaDr().IsPair()) {
	      // Lambda define 
	      CastFrame(Frame)->
		Define(Ref.CAADr().SymbolName(),
		       new TLispLambda(Frame, Ref.CDADr(),
				       new TLispPair(new
						     TLispSymbol("begin"),
						     Ref.CDDr())));
	      return Ref.CAADr();
	    }
	    else {
	      // Normal define
	      AssertNull(Ref.CdDDr());
	      CastFrame(Frame)->Define(Ref.CADr().SymbolName(),
				       lisp_eval(Ref.CADDr(), Frame));
	      return Ref.CADr();
	    }
	  }
	  else if (*Name == "set!") {
	    AssertNull(Ref.CddDr());
	    (*CastFrame(Frame))(Ref.CADr().SymbolName()) = lisp_eval(Ref.CADDr(), Frame);
	    return Ref.CADr();
	  }
	  else if (*Name == "quote") {
	    AssertNull(Ref.CdDr());
	    return Ref.CADr();
	  }
	  else if (*Name == "if") {
	    if (Ref.CddDr().IsPair())
	      AssertNull(Ref.CDDDDr());
	    else AssertNull(Ref.CDDDr());
	    if (!lisp_eval(Ref.CADr(), Frame).IsTrue()) {
	      if (Ref.CDDDr().IsPair()) {
		// this was not properly tail-recursive:
		//return lisp_eval(Ref.Cadddr(), Frame);
		Ref = Ref.CADDDr();
		continue;
	      }
	      return LispNull;
	    }
	    // this was not properly tail-recursive:
	    //return lisp_eval(Ref.Caddr(), Frame);
	    Ref = Ref.CADDr();
	    continue;
	  }
	  else if (*Name == "and") {
	    TLispRefOS List(Ref.CDr());
	    TLispRefOS Result(LispTrue);
	    while (List.IsPair() && !List.CDr().IsNull()) {
	      Result = lisp_eval(List.CAr(), Frame);	      
	      if (!Result.IsTrue()) return Result;
	      List = List.CDr();
	    }
	    Ref = List.Car();
	    //return Result;
	    continue;
	  }
	  else if (*Name == "or") {
	    TLispRefOS List(Ref.CDr());
	    TLispRefOS Result(LispFalse);
	    while (List.IsPair() && !List.CDr().IsNull()) {
	      Result = lisp_eval(List.CAr(), Frame);	      
	      if (Result.IsTrue()) return Result;
	      List = List.CDr();
	    }
	    Ref = List.Car();
	    //return Result;
	    continue;
	  }
	  else if (*Name == "begin") {
	    TLispRefOS List(Ref.CDr());
	    if (List.IsNull()) return LispNull;
	    //TLispRef Result = LispNull;
	    while (List.IsPair() && !List.CDr().IsNull()) {
	      //Result =
	      lisp_eval(List.CAr(), Frame);	      
	      List = List.CDr();
	    }
	    // this was not properly tail-recursive
	    // return Result;
	    Ref = List.Car();
	    continue;
	  }
	  else if (*Name == "lambda") {
	    AssertNull(Ref.CddDr());
	    return new TLispLambda(Frame, Ref.CADr(), Ref.CADDr());
	  }
	  else if (*Name == "eval") {
	    TLispRefOS EvalFrame(Ref.CdDr().IsNull()
				 ? Frame : lisp_eval(Ref.CADDr(), Frame));
	    // this was not properly tail-recursive:
	    //return lisp_eval(lisp_eval(Ref.Cadr(), Frame), EvalFrame);
	    Ref = lisp_eval(Ref.CADr(), Frame);
	    Frame = EvalFrame;
	    continue;
	  }
	  else {
	    // Handle user syntaxes
	    TLispRef *P = CastFrame(Frame)->find(*SyntaxAListName);
	    if (P) {
	      TLispRefOS SyntaxAList(*P);
	      while (SyntaxAList.IsPair()) {
		if (SyntaxAList.CaAr().SymbolName() == *Name) break;
		// was not properly tail-recursive
		//return SyntaxAList.Cdar().Apply(new TLispPair(Frame, Ref.Cdr()));
		SyntaxAList = SyntaxAList.CDr();
	      }
	      if (SyntaxAList.IsPair()) {
		if (SyntaxAList.CdAr().IsLambda()) {
		  CastLambda(SyntaxAList.CDAr())->
		    ApplyAux(new TLispPair(Frame, Ref.CDr()),
			     /*returned*/Ref, /*returned*/Frame);
		  continue;
		}
		else return SyntaxAList.CDAr().Apply(new TLispPair(Frame, Ref.CDr()));
	      }
	    }
	  }
	}
	// Evaluate operator and arguments
	TLispRefOS Evaluated
	  = TLispRef(new TLispPair(lisp_eval(Ref.CAr(), Frame),
				   LispNull)); 
	TLispRef *CurrentEvaluated = &Evaluated;
	TLispRefOS Current(Ref.CDr());
	while (Current.IsPair()) {
	  CurrentEvaluated = &(CurrentEvaluated->CDr() = new
			       TLispPair(lisp_eval(Current.CAr(), Frame),
					 LispNull));
	  Current = Current.CDr();
	}
	CurrentEvaluated->CDr() = lisp_eval(Current, Frame);
	// Apply operator to arguments
	if (Evaluated.CAr().IsLambda()) {
	  // special handling for proper tail-recursiveness
	  TLispRefOS ArgList(Evaluated.CDr());
	  CastLambda(Evaluated.CAr())->
	    ApplyAux(ArgList, /*returned*/Ref, /*returned*/Frame);
	  continue;
	}
	else return Evaluated.CAr().Apply(Evaluated.CDr());
      }
    default:
      // TLispBase::Null:
      // TLispBase::Number:
      // TLispBase::String:
      // TLispBase::Eof:
      // TLispBase::Char:
      // TLispBase::Boolean:
      // self-evaluating
      return Ref;
    }
  }
}

#if defined(OWNEXCEPTIONS)

XLisp *Exception;
TTry *Tries;
TLispRefOS *TLispRefOS::First;

TTry **FindTryOfThread()
{
  void *mystack;
  asm("movl %%esp, %%eax" : "=a" (mystack));
  TTry **Try;
  for (Try = &Tries; *Try; Try = &(*Try)->Next)
    if (mystack >= (*Try)->minstack && mystack <= (*Try)->maxstack)
      return Try;
  return 0;
}

void TryFollowing()
{
  TTry *Try = new TTry;
  // copy thread id
  TTry *TryOfThread = *FindTryOfThread();
  Try->minstack = TryOfThread->minstack;
  Try->maxstack = TryOfThread->maxstack;
  // fetch current continuation
  asm("leal 8(%%ebp), %%eax" : "=a" (Try->stack));
  asm("" : "=b" (Try->base));
  asm("movl (%%ebp), %%eax" : "=a" (Try->frame));
  asm("movl 4(%%ebp), %%eax" : "=a" (Try->continuation));
  Try->Next = Tries;
  Tries = Try;
}

void RegisterThreadWithTries(void *minstack, void *maxstack)
{
  TTry *Try = new TTry;
  Try->minstack = minstack;
  Try->maxstack = maxstack;
  Try->stack = Try->frame = Try->base = Try->continuation = 0;
  Try->Next = Tries;
  Tries = Try;
}

void UnregisterThreadWithTries()
{
  TTry **T;
  while ((T = FindTryOfThread()) != 0) {
    TTry *TT = *T;
    *T = TT->Next;
    delete TT;
  }
}

void ThrowException(XLisp *X)
{
  Exception = X;
  TTry **T = FindTryOfThread();
  TTry Try;
  if (T) {
    TTry *TT = *T;
    Try = *TT; // copy
    *T = Try.Next;
    delete TT;
  } else {
#if defined(_LowLevel)
    kprintf("ThrowException failure");
    sys_exit_process();
#else
    printf("ThrowException failure");
    _exit(0);
#endif
  }

  void *CurStack;
  asm("mov %%esp,%%eax" : "=a" (CurStack):);
  TLispRefOS::KillAllBetween(CurStack, Try.stack);

  asm("movl %%edx, %%ebp\n movl %%ecx, %%esp\n jmpl %%eax"
      :
      : "d" (Try.frame), "c" (Try.stack), "b" (Try.base), "a" (Try.continuation));
}

#endif
