;;; bpascal.el --- major mode for editing pascal source in Emacs

;;; Modified for Borland Pascal specials
;;; by Matthias K"oppe <mkoeppe@csmd.cs.uni-magdeburg.de>
;;; 22-Sep-1997 "

;; Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.

;; Author: Espen Skoglund (espensk@stud.cs.uit.no)
;; Keywords: languages

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; USAGE
;; =====

;; Emacs should enter Bpascal mode when you find a Bpascal source file.
;; When you have entered Bpascal mode, you may get more info by pressing
;; C-h m. You may also get online help describing various functions by:
;; C-h f <Name of function you want described>

;; If you want to customize Bpascal mode to fit you better, you may add
;; these lines (the values of the variables presented here are the defaults):
;;
;; ;; User customization for Bpascal mode
;; (setq bpascal-indent-level       3
;;       bpascal-case-indent        2
;;       bpascal-auto-newline       nil
;;       bpascal-tab-always-indent  t
;;       bpascal-auto-endcomments   t
;;       bpascal-auto-lineup        '(all)
;;       bpascal-toggle-completions nil
;;       bpascal-type-keywords      '("array" "file" "packed" "char" 
;; 				      "integer" "real" "string" "record")
;;       bpascal-start-keywords     '("begin" "end" "function" "procedure"
;; 				      "repeat" "until" "while" "read" "readln"
;; 				      "reset" "rewrite" "write" "writeln")
;;       bpascal-separator-keywords '("downto" "else" "mod" "div" "then"))

;; KNOWN BUGS / BUGREPORTS
;; =======================
;; As far as I know, there are no bugs in the current version of this
;; package.  This may not be true however, since I never use this mode
;; myself and therefore would never notice them anyway.   If you do
;; find any bugs, you may submit them to: espensk@stud.cs.uit.no
;; as well as to bug-gnu-emacs@prep.ai.mit.edu.

;;; Code:

(defvar bpascal-mode-abbrev-table nil
  "Abbrev table in use in Bpascal-mode buffers.")
(define-abbrev-table 'bpascal-mode-abbrev-table ())

(defvar bpascal-mode-map ()
  "Keymap used in Bpascal mode.")
(if bpascal-mode-map
    ()
  (setq bpascal-mode-map (make-sparse-keymap))
  (define-key bpascal-mode-map ";"        'electric-bpascal-semi-or-dot)
  (define-key bpascal-mode-map "."        'electric-bpascal-semi-or-dot)
  (define-key bpascal-mode-map ":"        'electric-bpascal-colon)
  (define-key bpascal-mode-map "="        'electric-bpascal-equal)
  (define-key bpascal-mode-map "#"        'electric-bpascal-hash)
  (define-key bpascal-mode-map "\r"       'electric-bpascal-terminate-line)
  (define-key bpascal-mode-map "\t"       'electric-bpascal-tab)
  (define-key bpascal-mode-map "\M-\t"    'bpascal-complete-word)
;  (define-key bpascal-mode-map "\M-?"     'bpascal-show-completions)
  (define-key bpascal-mode-map "\177"     'backward-delete-char-untabify)
  (define-key bpascal-mode-map "\M-\C-h"  'bpascal-mark-defun)
  (define-key bpascal-mode-map "\C-c\C-b" 'bpascal-insert-block)
  (define-key bpascal-mode-map "\M-*"     'bpascal-star-comment)
  (define-key bpascal-mode-map "\C-c\C-c" 'bpascal-comment-area)
  (define-key bpascal-mode-map "\C-c\C-u" 'bpascal-uncomment-area)
  (define-key bpascal-mode-map "\M-\C-a"  'bpascal-beg-of-defun)
  (define-key bpascal-mode-map "\M-\C-e"  'bpascal-end-of-defun)
  (define-key bpascal-mode-map "\C-c\C-d" 'bpascal-goto-defun)
  (define-key bpascal-mode-map "\C-c\C-o" 'bpascal-outline)
  (define-key bpascal-mode-map "\C-C\C-i" 'bpascal-make-imenu)
;;; A command to change the whole buffer won't be used terribly
;;; often, so no need for a key binding.
;  (define-key bpascal-mode-map "\C-cd"    'bpascal-downcase-keywords)
;  (define-key bpascal-mode-map "\C-cu"    'bpascal-upcase-keywords)
;  (define-key bpascal-mode-map "\C-cc"    'bpascal-capitalize-keywords)
  )

(defvar bpascal-imenu-generic-expression
  '(("Classes"
     "[ \t\n]+\\([a-zA-Z0-9_]+\\)[ \t]*=[ \t]*\\(object\\|record\\)" 1)
    ("Methods"
     "^[ \t]*\\(function\\|procedure\\|constructor\\|destructor\\)[ \t\n]+\\([a-zA-Z0-9_]+\\.[a-zA-Z0-9_]+\\)" 2)
    ("Functions"
     "^[ \t]*\\(function\\|procedure\\)[ \t\n]+\\([a-zA-Z0-9_]+\\)" 2)
    ("Sections"
     "[ \t\n]+\\(interface\\|implementation\\)[ \t\n]+" 1)
    ("Includes"
     "\\{$[iI][ \t]+\\([-a-zA-Z0-9_.+]+\\)[ \t]*}" 1)
    ("Conditionals"
     "\\{$ifn?def[ \t]+\\([a-zA-Z0-9_]+\\)[ \t]*}" 1))
  
  "Imenu expression for Bpascal-mode.  See `imenu-generic-expression'.")
  
(defvar bpascal-keywords
  '("and" "array" "begin" "case" "const" "div" "do" "downto" "else" "end" 
    "file" "for" "function" "goto" "if" "in" "label" "mod" "nil" "not" "of" 
    "or" "packed" "procedure" "program" "record" "repeat" "set" "then" "to" 
    "type" "until" "var" "while" "with"
    ;; The following are Borland specific
    "xor" "constructor" "destructor" "object" "private" "public" "asm"
    "exports" "implementation" "inherited" "inline" "interface" "library"
    "shl" "shr" "string" "unit" "uses" 
    ;; The following are not standard in pascal, but widely used.
    ;"get" "put" "input" "output" "read" "readln" "reset" "rewrite" "write"
    ;"writeln"
    ))

;;;
;;; Regular expressions used to calculate indent, etc.
;;;
(defconst bpascal-symbol-re      "\\<[a-zA-Z_][a-zA-Z_0-9.]*\\>")
(defconst bpascal-beg-block-re   "\\<\\(begin\\|asm\\|case\\|record\\|object\\|repeat\\)\\>")
(defconst bpascal-end-block-re   "\\<\\(end\\|until\\)\\>")
(defconst bpascal-declaration-re "\\<\\(const\\|label\\|type\\|var\\)\\>")
(defconst bpascal-defun-re       "\\<\\(function\\|procedure\\|constructor\\|destructor\\|unit\\|library\\|program\\)\\>")
(defconst bpascal-sub-block-re   "\\<\\(if\\|else\\|for\\|while\\|with\\)\\>")
(defconst bpascal-noindent-re    "\\<\\(begin\\|end\\|until\\|else\\)\\>")
(defconst bpascal-nosemi-re      "\\<\\(begin\\|repeat\\|then\\|do\\|else\\)\\>")
(defconst bpascal-autoindent-lines-re
  "\\<\\(label\\|var\\|type\\|const\\|until\\|end\\|begin\\|repeat\\|else\\)\\>")
(defconst bpascal-section-re     "\\<\\(interface\\|implementation\\)\\>")

;;; Strings used to mark beginning and end of excluded text
(defconst bpascal-exclude-str-start "{---")
(defconst bpascal-exclude-str-end " ---}")

(defvar bpascal-mode-syntax-table nil
  "Syntax table in use in Bpascal-mode buffers.")

(if bpascal-mode-syntax-table
    ()
  (setq bpascal-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\\ "."   bpascal-mode-syntax-table)
  (modify-syntax-entry ?( "()1"  bpascal-mode-syntax-table)  
  (modify-syntax-entry ?) ")(4"  bpascal-mode-syntax-table)
  (modify-syntax-entry ?* ". 23" bpascal-mode-syntax-table)
  (modify-syntax-entry ?{ "<"    bpascal-mode-syntax-table)
  (modify-syntax-entry ?} ">"    bpascal-mode-syntax-table)
  (modify-syntax-entry ?+ "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?- "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?= "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?% "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?< "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?> "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?& "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?| "."    bpascal-mode-syntax-table)
  (modify-syntax-entry ?_ "w"    bpascal-mode-syntax-table)
  (modify-syntax-entry ?\' "\""  bpascal-mode-syntax-table))

(defvar bpascal-font-lock-keywords
  (list
   '("\\(#\\(\$[0-9a-fA-F]+\\|[0-9]+\\)\\)"
     (1 font-lock-string-face))
   '("^[ \t]*\\(function\\|constructor\\|destructor\\|unit\\|pro\\(cedure\\|gram\\)\\)\\>[ \t]*\\([A-Za-z0-9_.]+\\)?"
     (1 font-lock-keyword-face) (3 font-lock-function-name-face nil t))
;   ("type" "const" "real" "integer" "char" "boolean" "var"
;    "record" "array" "file")
   (cons (concat "\\<\\(array\\|b\\(yte\\|oolean\\)\\|c\\(har\\|onst\\)\\|file\\|"
		 "in\\(herited\\|teger\\)\\|longint\\|nil\\|object\\|pointer\\|re\\(al\\|cord\\)\\|"
		 "s\\(et\\|tring\\|hortint\\)\\|type\\|uses\\|var\\|word\\)\\>")
	 'font-lock-type-face)
   '("\\<\\(assembler\\|cdecl\\|label\\|external\\|f\\(ar\\|orward\\)\\|i\\(nterface\\|mplementation\\)\\|near\\|p\\(rivate\\|ublic\\)\\|virtual\\)\\>" . font-lock-reference-face)
   '("\\<\\([0-9]+\\)[ \t]*:" 1 font-lock-reference-face)
;   ("of" "to" "for" "if" "then" "else" "case" "while"
;    "do" "until" "and" "or" "not" "in" "with" "repeat" "begin" "end")
   (concat "\\<\\("
	   "a\\(sm\\|nd\\)\\|begin\\|case\\|d\\(iv\\|o\\)\\|e\\(lse\\|nd\\)\\|for\\|i[fn]\\|mod\\|"
	   "not\\|o\\([fr]\\|bject\\)\\|packed\\|re\\(cord\\|peat\\)\\|sh\\(l\\|r\\)\\|t\\(hen\\|o\\)\\|until\\|w\\(hile\\|ith\\)\\|xor"
	   "\\)\\>")
   '("\\<\\(goto\\)\\>[ \t]*\\([0-9]+\\)?"
     (1 font-lock-keyword-face) (2 font-lock-reference-face nil t)))
  "Additional expressions to highlight in Bpascal mode.")

(defvar bpascal-indent-level 2
  "*Indentation of Bpascal statements with respect to containing block.")

(defvar bpascal-case-indent 2
  "*Indentation for case statements.")

(defvar bpascal-auto-newline nil
  "*Non-nil means automatically newline after semicolons and the punctuation
mark after an end.")

(defvar bpascal-tab-always-indent t
  "*Non-nil means TAB in Bpascal mode should always reindent the current line,
regardless of where in the line point is when the TAB command is used.")

(defvar bpascal-auto-endcomments nil
  "*Non-nil means a comment { ... } is set after the ends which ends cases and
functions. The name of the function or case will be set between the braces.")

(defvar bpascal-auto-lineup '(paramlist declaration)
  "*List of contexts where auto lineup of :'s or ='s should be done.
Elements can be of type: 'paramlist', 'declaration' or 'case', which will
do auto lineup in parameterlist, declarations or case-statements
respectively. The word 'all' will do all lineups. '(case paramlist) for
instance will do lineup in case-statements and parameterlist, while '(all)
will do all lineups.")

(defvar bpascal-toggle-completions nil
  "*Non-nil means \\<bpascal-mode-map>\\[bpascal-complete-word] should try all possible completions one by one.
Repeated use of \\[bpascal-complete-word] will show you all of them.
Normally, when there is more than one possible completion,
it displays a list of all possible completions.")

(defvar bpascal-type-keywords
  '("array" "object" "file" "packed" "char" "integer" "real" "string" "record")
  "*Keywords for types used when completing a word in a declaration or parmlist.
\(eg. integer, real, char.)  The types defined within the Bpascal program
will be completed runtime, and should not be added to this list.")

(defvar bpascal-start-keywords
  '("begin" "asm" "end" "function" "procedure" "destructor" "constructor" "repeat" "until" "while"
    "read" "readln" "reset" "rewrite" "write" "writeln")
  "*Keywords to complete when standing at the first word of a statement.
\(eg. begin, repeat, until, readln.)
The procedures and variables defined within the Bpascal program
will be completed runtime and should not be added to this list.")

(defvar bpascal-separator-keywords
  '("downto" "else" "mod" "div" "then" "and" "or" "xor")
  "*Keywords to complete when NOT standing at the first word of a statement.
\(eg. downto, else, mod, then.) 
Variables and function names defined within the
Bpascal program are completed runtime and should not be added to this list.")

;;;
;;;  Macros
;;;

(defsubst bpascal-get-beg-of-line (&optional arg)
  (save-excursion
    (beginning-of-line arg)
    (point)))

(defsubst bpascal-get-end-of-line (&optional arg)
  (save-excursion
    (end-of-line arg)
    (point)))

(defun bpascal-declaration-end ()
  (let ((nest 1))
    (while (and (> nest 0)
		(re-search-forward 
		 "[:=]\\|\\(\\<\\(record\\|object\\)\\>\\)\\|\\(\\<end\\>\\)" 
		 (save-excursion (end-of-line 2) (point)) t))
      (cond ((match-beginning 1) (setq nest (1+ nest)))
	    ((match-beginning 2) (setq nest (1- nest)))
	    ((looking-at "[^(\n
]+)") (setq nest 0))))))


(defun bpascal-declaration-beg ()
  (let ((nest 1))
    (while (and (> nest 0)
		(re-search-backward "[:=]\\|\\<\\(type\\|var\\|label\\|const\\)\\>\\|\\(\\<\\(record\\|object\\)\\>\\)\\|\\(\\<end\\>\\)" (bpascal-get-beg-of-line 0) t))
      (cond ((match-beginning 1) (setq nest 0))
	    ((match-beginning 2) (setq nest (1- nest)))
	    ((match-beginning 3) (setq nest (1+ nest)))))
    (= nest 0)))

  
(defsubst bpascal-within-string ()
  (save-excursion
    (nth 3 (parse-partial-sexp (bpascal-get-beg-of-line) (point)))))

;;;

(defun make-dosish-line-ends ()
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward "[ \t
]*\n" nil t)
    (replace-match "
\n" nil nil)))

(defun make-unixish-line-ends ()
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward "[ \t
]+$" nil t)
    (replace-match "" nil nil)))

(defun bpascal-clean-line-ends ()
  (let ((modp (buffer-modified-p)))
    (make-unixish-line-ends)
    (goto-char (point-min))
    (set-buffer-modified-p modp)))

;;;###autoload
(defun bpascal-mode ()
  "Major mode for editing Bpascal code. \\<bpascal-mode-map>
TAB indents for Bpascal code.  Delete converts tabs to spaces as it moves back.

\\[bpascal-complete-word] completes the word around current point with respect \
to position in code
\\[bpascal-show-completions] shows all possible completions at this point.

Other useful functions are:

\\[bpascal-mark-defun]\t- Mark function.
\\[bpascal-insert-block]\t- insert begin ... end;
\\[bpascal-star-comment]\t- insert (* ... *)
\\[bpascal-comment-area]\t- Put marked area in a comment, fixing nested comments.
\\[bpascal-uncomment-area]\t- Uncomment an area commented with \
\\[bpascal-comment-area].
\\[bpascal-beg-of-defun]\t- Move to beginning of current function.
\\[bpascal-end-of-defun]\t- Move to end of current function.
\\[bpascal-goto-defun]\t- Goto function prompted for in the minibuffer.
\\[bpascal-outline]\t- Enter bpascal-outline-mode (see also bpascal-outline).

Variables controlling indentation/edit style:

 bpascal-indent-level      (default 3)
    Indentation of Bpascal statements with respect to containing block.
 bpascal-case-indent       (default 2)
    Indentation for case statements.
 bpascal-auto-newline      (default nil)
    Non-nil means automatically newline after semicolons and the punctuation
    mark after an end.
 bpascal-tab-always-indent (default t)
    Non-nil means TAB in Bpascal mode should always reindent the current line,
    regardless of where in the line point is when the TAB command is used.
 bpascal-auto-endcomments  (default t)
    Non-nil means a comment { ... } is set after the ends which ends cases and
    functions. The name of the function or case will be set between the braces.
 bpascal-auto-lineup       (default t)
    List of contexts where auto lineup of :'s or ='s hould be done.

See also the user variables bpascal-type-keywords, bpascal-start-keywords and
bpascal-separator-keywords.

Turning on Bpascal mode calls the value of the variable bpascal-mode-hook with
no args, if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map bpascal-mode-map)
  (setq major-mode 'bpascal-mode)
  (setq mode-name "Bpascal")
  (setq local-abbrev-table bpascal-mode-abbrev-table)
  (set-syntax-table bpascal-mode-syntax-table)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'bpascal-indent-line)
  (setq comment-indent-function 'bpascal-indent-comment)
  (make-local-variable 'parse-sexp-ignore-comments)
  (setq parse-sexp-ignore-comments nil)
  (make-local-variable 'case-fold-search)
  (setq case-fold-search t)
  (make-local-variable 'comment-start)
  (setq comment-start "{")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "(\\*+ *\\|{ *")
  (make-local-variable 'comment-end)
  (setq comment-end "}")
  ;; Font lock support
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(bpascal-font-lock-keywords nil t))
  ;; Imenu support
  (make-local-variable 'imenu-generic-expression)
  (setq imenu-generic-expression bpascal-imenu-generic-expression)
  (font-lock-mode t)
;  (imenu-add-to-menubar "Constructs")

  (easy-menu-define bpascal-mode-menu
		    bpascal-mode-map
		    "Menu used in BPascal mode."
		    (list "Pascal"
			  ["Mark function" bpascal-mark-defun "M-C-h"]
			  ["Beginning of function" bpascal-beg-of-defun  "M-C-a"]
			  ["End of function" bpascal-end-of-defun  "M-C-e"]
			  ["Go to function..." bpascal-goto-defun  "C-c C-d"]
			  "-"
			  ["Insert block" bpascal-insert-block  "C-c C-b"]
			  ["Comment out" bpascal-comment-area  "C-c C-c"]
			  ["Comment in" bpascal-uncomment-area  "C-c C-u"]
			  "-"
			  ["Outline mode" bpascal-outline "C-c C-o"]))  

  (if (< emacs-major-version 20) (bpascal-clean-line-ends))
  (run-hooks 'bpascal-mode-hook))



;;;
;;;  Electric functions
;;;
(defun electric-bpascal-terminate-line ()
  "Terminate line and indent next line."
  (interactive)
  ;; First, check if current line should be indented
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (if (looking-at bpascal-autoindent-lines-re)
	(bpascal-indent-line)))
  (delete-horizontal-space) ; Removes trailing whitespaces
  (newline)
  ;; Indent next line
  (bpascal-indent-line)
  ;; Maybe we should set some endcomments
  (if bpascal-auto-endcomments
      (bpascal-set-auto-comments))
  ;; Check if we shall indent inside comment
  (let ((setstar nil))
    (save-excursion
      (forward-line -1)
      (skip-chars-forward " \t")
      (cond ((looking-at "\\*[ \t]+)")
	     ;; Delete region between `*' and `)' if there is only whitespaces.
	     (forward-char 1)
	     (delete-horizontal-space))
	    ((and (looking-at "(\\*\\|\\*[^)]")
		  (not (save-excursion
			 (search-forward "*)" (bpascal-get-end-of-line) t))))
	     (setq setstar t))))
    ;; If last line was a star comment line then this one shall be too.
    (if (null setstar)	
	(bpascal-indent-line)
      (insert "*  "))))


(defun electric-bpascal-semi-or-dot ()
  "Insert `;' or `.' character and reindent the line."
  (interactive)
  (insert last-command-char)
  (save-excursion
    (beginning-of-line)
    (bpascal-indent-line))
  (if bpascal-auto-newline
      (electric-bpascal-terminate-line)))

(defun electric-bpascal-colon ()
  "Insert `:' and do all indentions except line indent on this line."
  (interactive)
  (insert last-command-char)
  ;; Do nothing if within string.
  (if (bpascal-within-string)
      ()
    (save-excursion
      (beginning-of-line)
      (bpascal-indent-line))
    (let ((bpascal-tab-always-indent nil))
      (bpascal-indent-command))))

(defun electric-bpascal-equal ()
  "Insert `=', and do indention if within type declaration."
  (interactive)
  (insert last-command-char)
  (if (eq (car (bpascal-calculate-indent)) 'declaration)
      (let ((bpascal-tab-always-indent nil))
	(bpascal-indent-command))))

(defun electric-bpascal-hash ()
  "Insert `#', and indent to column 0 if this is a CPP directive."
  (interactive)
  (insert last-command-char)
  (if (save-excursion (beginning-of-line) (looking-at "^[ \t]*#"))
      (save-excursion (beginning-of-line)
		      (delete-horizontal-space))))

(defun electric-bpascal-tab ()
  "Function called when TAB is pressed in Bpascal mode."
  (interactive)
  ;; Do nothing if within a string or in a CPP directive.
  (if (or (bpascal-within-string)
	  (and (not (bolp))
	       (save-excursion (beginning-of-line) (eq (following-char) ?#))))
      (insert "\t")
    ;; If bpascal-tab-always-indent, indent the beginning of the line.
    (if bpascal-tab-always-indent
	(save-excursion
	  (beginning-of-line)
	  (bpascal-indent-line))
      (if (save-excursion
	    (skip-chars-backward " \t")
	    (bolp))
	  (bpascal-indent-line)
	(insert "\t")))
    (bpascal-indent-command)))



;;;
;;; Interactive functions
;;;
(defun bpascal-insert-block ()
  "Insert Bpascal begin ... end; block in the code with right indentation."
  (interactive)
  (bpascal-indent-line)
  (insert "begin")
  (electric-bpascal-terminate-line)
  (save-excursion
    (electric-bpascal-terminate-line)
    (insert "end;")
    (beginning-of-line)
    (bpascal-indent-line)))

(defun bpascal-star-comment ()
  "Insert Bpascal star comment at point."
  (interactive)
  (bpascal-indent-line)
  (insert "(*")
  (electric-bpascal-terminate-line)
  (save-excursion
    (electric-bpascal-terminate-line)
    (delete-horizontal-space)
    (insert ")"))
  (insert "  "))

(defun bpascal-mark-defun ()
  "Mark the current bpascal function (or procedure).
This puts the mark at the end, and point at the beginning."
  (interactive)
  (push-mark (point))
  (bpascal-end-of-defun)
  (push-mark (point))
  (bpascal-beg-of-defun)
  (if (fboundp 'zmacs-activate-region)
      (zmacs-activate-region)))

(defun bpascal-comment-area (start end)
  "Put the region into a Bpascal comment.
The comments that are in this area are \"deformed\":
`*)' becomes `!(*' and `}' becomes `!{'.
These deformed comments are returned to normal if you use
\\[bpascal-uncomment-area] to undo the commenting.

The commented area starts with `bpascal-exclude-str-start', and ends with
`bpascal-include-str-end'.  But if you change these variables,
\\[bpascal-uncomment-area] won't recognize the comments."
  (interactive "r")
  (save-excursion
    ;; Insert start and endcomments
    (goto-char end)
    (if (and (save-excursion (skip-chars-forward " \t") (eolp))
	     (not (save-excursion (skip-chars-backward " \t") (bolp))))
	(forward-line 1)
      (beginning-of-line))
    (insert bpascal-exclude-str-end)
    (setq end (point))
    (newline)
    (goto-char start)
    (beginning-of-line)
    (insert bpascal-exclude-str-start)
    (setq start (point)) ;;
    (newline)
    ;; Replace end-comments within commented area
    (goto-char end)
    (save-excursion
      (while (re-search-backward "\\*)" start t)
	(replace-match "$)$" t t)))
    (save-excursion
      (while (re-search-backward "\\*(" start t)
	(replace-match "$($" t t)))
    (save-excursion
      (while (re-search-backward "{" start t)
	(replace-match "%(%" t t)))
    (save-excursion
      (while (re-search-backward "}" start t)
	(replace-match "%)%" t t)))))

(defun bpascal-uncomment-area ()
  "Uncomment a commented area; change deformed comments back to normal.
This command does nothing if the pointer is not in a commented
area.  See also `bpascal-comment-area'."
  (interactive)
  (save-excursion
    (let ((start (point))
	  (end (point)))
      ;; Find the boundaries of the comment
      (save-excursion
	(setq start (progn (search-backward bpascal-exclude-str-start nil t)
			   (point)))
	(setq end (progn (search-forward bpascal-exclude-str-end nil t)
			 (point))))
      ;; Check if we're really inside a comment
      (if (or (equal start (point)) (<= end (point)))
	  (message "Not standing within commented area.")
	(progn
	  ;; Remove endcomment
	  (goto-char end)
	  (beginning-of-line)
	  (let ((pos (point)))
	    (end-of-line)
	    (delete-region pos (1+ (point))))
	  ;; Change comments back to normal
	  (save-excursion
	    (while (re-search-backward "%(%" start t)
	      (replace-match "{" t t)))
	  (save-excursion
	    (while (re-search-backward "%)%" start t)
	      (replace-match "}" t t)))
	  (save-excursion
	    (while (re-search-backward "$($" start t)
	      (replace-match "(*" t t)))
	  (save-excursion
	    (while (re-search-backward "$)$" start t)
	      (replace-match "*)" t t)))
	  ;; Remove startcomment
	  (goto-char start)
	  (beginning-of-line)
	  (let ((pos (point)))
	    (end-of-line)
	    (delete-region pos (1+ (point)))))))))

(defun bpascal-beg-of-defun ()
  "Move backward to the beginning of the current function or procedure."
  (interactive)
  (catch 'found
    (if (not (looking-at (concat "\\s \\|\\s)\\|" bpascal-defun-re)))
	(forward-sexp 1))
    (let ((nest 0) (max -1) (func 0)
	  (reg (concat bpascal-beg-block-re "\\|" 
		       bpascal-end-block-re "\\|"
		       bpascal-defun-re)))
      (while (re-search-backward reg nil 'move)
	(cond ((let ((state (save-excursion
			      (parse-partial-sexp (point-min) (point)))))
		 (or (nth 3 state) (nth 4 state))) ; Inside string or comment
	       ())
	      ((match-end 1)                       ; begin|case|record|repeat
	       (if (and (looking-at "\\<\\(record\\|object\\)\\>") (>= max 0))
		   (setq func (1- func)))
	       (setq nest (1+ nest)
		     max (max nest max)))
	      ((match-end 2)                       ; end|until
	       (if (and (= nest max) (>= max 0))
		   (setq func (1+ func)))
	       (setq nest (1- nest)))
	      ((match-end 3)                       ; function|procedure
	       (if (= 0 func)
		   (throw 'found t)
		 (setq func (1- func)))))))
    nil))

(defun bpascal-end-of-defun ()
  "Move forward to the end of the current function or procedure."
  (interactive)
  (if (looking-at "\\s ")
      (forward-sexp 1))
  (if (not (looking-at bpascal-defun-re))
      (bpascal-beg-of-defun))
  (forward-char 1)
  (let ((nest 0) (func 1)
	(reg (concat bpascal-beg-block-re "\\|" 
		     bpascal-end-block-re "\\|"
		     bpascal-defun-re)))
    (while (and (/= func 0)
		(re-search-forward reg nil 'move))
      (cond ((let ((state (save-excursion
			      (parse-partial-sexp (point-min) (point)))))
		 (or (nth 3 state) (nth 4 state))) ; Inside string or comment
	       ())
	    ((match-end 1)
	     (setq nest (1+ nest))
	     (if (save-excursion
		   (goto-char (match-beginning 0))
		   (looking-at "\\<\\(record\\|object\\)\\>"))
		 (setq func (1+ func))))
	    ((match-end 2)
	     (setq nest (1- nest))
	     (if (= nest 0)
		 (setq func (1- func))))
	    ((match-end 3)
	     (setq func (1+ func))))))
  (forward-line 1))

(defun bpascal-end-of-statement ()
  "Move forward to end of current statement."
  (interactive)
  (let ((parse-sexp-ignore-comments t)
	(nest 0) pos
	(regexp (concat "\\(" bpascal-beg-block-re "\\)\\|\\("
			bpascal-end-block-re "\\)")))
    (if (not (looking-at "[ \t\n
]")) (forward-sexp -1))
    (or (looking-at bpascal-beg-block-re)
	;; Skip to end of statement
	(setq pos (catch 'found
		    (while t
		      (forward-sexp 1)
		      (cond ((looking-at "[ \t]*;")
			     (skip-chars-forward "^;")
			     (forward-char 1)
			     (throw 'found (point)))
			    ((save-excursion
			       (forward-sexp -1)
			       (looking-at bpascal-beg-block-re))
			     (goto-char (match-beginning 0))
			     (throw 'found nil))
			    ((eobp)
			     (throw 'found (point))))))))
    (if (not pos)
	;; Skip a whole block
	(catch 'found
	  (while t
	    (re-search-forward regexp nil 'move)
	    (setq nest (if (match-end 1) 
			   (1+ nest)
			 (1- nest)))
	    (cond ((eobp)
		   (throw 'found (point)))
		  ((= 0 nest)
		   (throw 'found (bpascal-end-of-statement))))))
      pos)))

(defun bpascal-downcase-keywords ()
  "Downcase all Bpascal keywords in the buffer."
  (interactive)
  (bpascal-change-keywords 'downcase-word))

(defun bpascal-upcase-keywords ()
  "Upcase all Bpascal keywords in the buffer."
  (interactive)
  (bpascal-change-keywords 'upcase-word))

(defun bpascal-capitalize-keywords ()
  "Capitalize all Bpascal keywords in the buffer."
  (interactive)
  (bpascal-change-keywords 'capitalize-word))

;; Change the keywords according to argument.
(defun bpascal-change-keywords (change-word)
  (save-excursion
    (let ((keyword-re (concat "\\<\\("
			      (mapconcat 'identity bpascal-keywords "\\|")
			      "\\)\\>")))
      (goto-char (point-min))
      (while (re-search-forward keyword-re nil t)
	(funcall change-word -1)))))



;;;
;;; Other functions
;;;
(defun bpascal-set-auto-comments ()
  "Insert `{ case }' or `{ NAME }' on this line if appropriate.
Insert `{ case }' if there is an `end' on the line which
ends a case block.  Insert `{ NAME }' if there is an `end'
on the line which ends a function or procedure named NAME."
  (save-excursion
    (forward-line -1)
    (skip-chars-forward " \t")
    (if (and (looking-at "\\<end;")
	     (not (save-excursion
		    (end-of-line)
		    (search-backward "{" (bpascal-get-beg-of-line) t))))
	(let ((type (car (bpascal-calculate-indent))))
	  (if (eq type 'declaration)
	      ()
	    (if (eq type 'case)
		;; This is a case block
		(progn
		  (end-of-line)
		  (delete-horizontal-space)
		  (insert " { case }"))
	      (let ((nest 1))
		;; Check if this is the end of a function
		(save-excursion
		  (while (not (or (looking-at bpascal-defun-re) (bobp)))
		    (backward-sexp 1)
		    (cond ((looking-at bpascal-beg-block-re)
			   (setq nest (1- nest)))
			  ((looking-at bpascal-end-block-re)
			   (setq nest (1+ nest)))))
		  (if (bobp)
		      (setq nest 1)))
		(if (zerop nest)
		    (progn
		      (end-of-line)
		      (delete-horizontal-space)
		      (insert " { ")
		      (let (b e)
			(save-excursion
			  (setq b (progn (bpascal-beg-of-defun)
					 (skip-chars-forward "^ \t")
					 (skip-chars-forward " \t")
					 (point))
				e (progn (skip-chars-forward "a-zA-Z0-9_.") ;;
					 (point))))
			(insert-buffer-substring (current-buffer) b e))
		      (insert " }"))))))))))



;;;
;;; Indentation
;;;
(defconst bpascal-indent-alist
  '((block . (+ ind bpascal-indent-level))
    (case . (+ ind bpascal-case-indent))
    (caseblock . ind) (cpp . 0)
    (declaration . (+ ind bpascal-indent-level))
    (paramlist . (bpascal-indent-paramlist t))
    (comment . (bpascal-indent-comment t))
    (section . 0)
    (defun . ind) (contexp . ind)
    (unknown . 0) (string . 0)))

(defun bpascal-indent-command ()
  "Indent for special part of code."
  (let* ((indent-str (bpascal-calculate-indent))
	 (type (car indent-str))
	 (ind (car (cdr indent-str))))
    (cond ((and (eq type 'paramlist)
		(or (memq 'all bpascal-auto-lineup)
		    (memq 'paramlist bpascal-auto-lineup)))
	   (bpascal-indent-paramlist)
	   (bpascal-indent-paramlist))
	  ((and (eq type 'declaration)
		(or (memq 'all bpascal-auto-lineup)
		    (memq 'declaration  bpascal-auto-lineup)))
	   (bpascal-indent-declaration))
	  ((and (eq type 'case) (not (looking-at "^[ \t]*$"))
		(or (memq 'all bpascal-auto-lineup)
		    (memq 'case bpascal-auto-lineup)))
	   (bpascal-indent-case)))
    (if (looking-at "[ \t]+$")
	(skip-chars-forward " \t"))))

(defun bpascal-indent-line ()
  "Indent current line as a Bpascal statement."
  (let* ((indent-str (bpascal-calculate-indent))
	 (type (car indent-str))
	 (ind (car (cdr indent-str))))
    (if (looking-at "^[0-9a-zA-Z]+[ \t]*:[^=]")
	(search-forward ":" nil t))
    (delete-horizontal-space)
    ;; Some things should not be indented
    (if (or (and (eq type 'declaration) (looking-at bpascal-declaration-re))
	    (eq type 'cpp)
	    (looking-at "{\\$")
;	    (looking-at bpascal-defun-re)
	    )
	()
      ;; Other things should have no extra indent
      (if (looking-at bpascal-noindent-re)
	  (indent-to ind)
	;; But most lines are treated this way:
	(indent-to (eval (cdr (assoc type bpascal-indent-alist))))
	))))

(defun bpascal-calculate-indent ()
  "Calculate the indent of the current Bpascal line.
Return a list of two elements: (INDENT-TYPE INDENT-LEVEL)."
  (save-excursion
    (let* ((parse-sexp-ignore-comments t)
	   (oldpos (point))
	   (state (save-excursion (parse-partial-sexp (point-min) (point))))
	   (nest 0) (par 0) (complete (looking-at "[ \t]*end\\>"))
	   (elsed (looking-at "[ \t]*else\\>"))
	   (type (catch 'nesting
		   ;; Check if inside a string, comment or parenthesis
		   (cond ((nth 3 state) (throw 'nesting 'string))
			 ((nth 4 state) (throw 'nesting 'comment))
			 ((> (car state) 0)
			  (goto-char (scan-lists (point) -1 (car state)))
			  (setq par (1+ (current-column))))
			 ((save-excursion (beginning-of-line)
					  (eq (following-char) ?#))
			  (throw 'nesting 'cpp)))
		   ;; Loop until correct indent is found
		   (while t
		     (backward-sexp 1)
		     (cond (;--Escape from case statements
			    (and (looking-at "[A-Za-z0-9]+[ \t]*:[^=]")
				 (not complete)
				 (save-excursion (skip-chars-backward " \t")
						 (bolp))
				 (= (save-excursion
				      (end-of-line) (backward-sexp) (point))
				    (point))
				 (> (save-excursion (goto-char oldpos)
						    (beginning-of-line)
						    (point))
				    (point)))
			    (throw 'nesting 'caseblock))
			   (; -- Start anew at each section
			    (looking-at bpascal-section-re)
			    (throw 'nesting 'unknown)
			    (setq nest 0))			    
			   (;--Nest block outwards
			    (looking-at bpascal-beg-block-re)
			    (if (= nest 0)
				(cond ((looking-at "case\\>")
				       (throw 'nesting 'case))
				      ((looking-at "\\(record\\|object\\)\\>")
				       (throw 'nesting 'declaration))
				      (t (throw 'nesting 'block)))
			      (setq nest (1- nest))))
			   (;--Nest block inwards
			    (looking-at bpascal-end-block-re)
			    (if (and (looking-at "end\\s ")
				     elsed (not complete))
				(throw 'nesting 'block))
			    (setq complete t
				  nest (1+ nest)))
			   (;--Defun (or parameter list)
			    (looking-at bpascal-defun-re)
			    (if (= 0 par)
				(throw 'nesting 'defun)
			      (setq par 0)
			      (let ((n 0))
				(while (re-search-forward
					"\\(\\<\\(record\\|object\\)\\>\\)\\|\\<end\\>"
					oldpos t)
				  (if (match-end 1)
				      (setq n (1+ n)) (setq n (1- n))))
				(if (> n 0)
				    (throw 'nesting 'declaration)
				  (throw 'nesting 'paramlist)))))
			   (;--Declaration part
			    (looking-at bpascal-declaration-re)
			    (if (save-excursion
				  (goto-char oldpos)
				  (forward-line -1)
				  (looking-at "^[ \t]*$"))
				(throw 'nesting 'unknown)
			      (throw 'nesting 'declaration)))
			   (;--If, else or while statement
			    (and (not complete)
				 (looking-at bpascal-sub-block-re))
			    (throw 'nesting 'block))
			   (;--Found complete statement
			    (save-excursion (forward-sexp 1)
					    (= (following-char) ?\;))
			    (setq complete t))
			   (;--No known statements
			    (bobp)
			    (throw 'nesting 'unknown))
			   )))))

      ;; Return type of block and indent level.
      (if (> par 0)                               ; Unclosed Parenthesis 
	  (list 'contexp par)
	(list type (bpascal-indent-level))))))

(defun bpascal-indent-level ()
  "Return the indent-level the current statement has.
Do not count labels, case-statements or records."
  (save-excursion
    (beginning-of-line)
    (if (looking-at "[ \t]*[0-9a-zA-Z]+[ \t]*:[^=]")
	(search-forward ":" nil t)
;      (if (looking-at ".*=[ \t]*\\(record\\|object\\)\\>")
;	  (search-forward "=" nil t))
)
    (skip-chars-forward " \t")
    (current-column)))

(defun bpascal-indent-comment (&optional arg)
  "Indent current line as comment.
If optional arg is non-nil, just return the
column number the line should be indented to."
    (let* ((stcol (save-excursion
		    (re-search-backward "(\\*\\|{" nil t)
		    (1+ (current-column)))))
      (if arg stcol
	(delete-horizontal-space)
	(indent-to stcol))))

(defun bpascal-indent-case ()
  "Indent within case statements."
  (let ((savepos (point-marker))
	(end (prog2
		 (end-of-line)
		 (point-marker)
	       (re-search-backward "\\<case\\>" nil t)))
	(beg (point)) oldpos
	(ind 0))
    ;; Get right indent
    (while (< (point) (marker-position end))
      (if (re-search-forward 
	   "^[ \t]*[^ \t,:]+[ \t]*\\(,[ \t]*[^ \t,:]+[ \t]*\\)*:"
	   (marker-position end) 'move)
	  (forward-char -1))
      (if (< (point) (marker-position end))
	  (progn
	    (delete-horizontal-space)
	    (if (> (current-column) ind)
		(setq ind (current-column)))
	    (bpascal-end-of-statement))))
    (goto-char beg)
    (setq oldpos (marker-position end))
    ;; Indent all case statements
    (while (< (point) (marker-position end))
      (if (re-search-forward
	   "^[ \t]*[^][ \t,\\.:]+[ \t]*\\(,[ \t]*[^ \t,:]+[ \t]*\\)*:"
	   (marker-position end) 'move)
	  (forward-char -1))
      (indent-to (1+ ind))
      (if (/= (following-char) ?:)
	  ()
	(forward-char 1)
	(delete-horizontal-space)
	(insert " "))
      (setq oldpos (point))
      (bpascal-end-of-statement))
    (goto-char savepos)))

(defun bpascal-indent-paramlist (&optional arg)
  "Indent current line in parameterlist.
If optional arg is non-nil, just return the
indent of the current line in parameterlist."
  (save-excursion
    (let* ((oldpos (point))
	   (stpos (progn (goto-char (scan-lists (point) -1 1)) (point)))
	   (stcol (1+ (current-column)))
	   (edpos (progn (bpascal-declaration-end) 
			 (search-backward ")" (bpascal-get-beg-of-line) t)
			 (point)))
	   (usevar (re-search-backward "\\<var\\>" stpos t)))
      (if arg (progn
		;; If arg, just return indent
		(goto-char oldpos)
		(beginning-of-line)
		(if (or (not usevar) (looking-at "[ \t]*var\\>"))
		    stcol (+ 4 stcol)))
	(goto-char stpos)
	(forward-char 1)
	(delete-horizontal-space)
	(if (and usevar (not (looking-at "var\\>")))
	    (indent-to (+ 4 stcol)))
	(bpascal-indent-declaration nil stpos edpos)))))

(defun bpascal-indent-declaration (&optional arg start end)
  "Indent current lines as declaration, lining up the `:'s or `='s."
  (let ((pos (point-marker)))
    (if (and (not (or arg start)) (not (bpascal-declaration-beg)))
	()
      (let ((lineup (if (or (looking-at "\\<var\\>\\|\\<\\(record\\|object\\)\\>") arg start) 
			":" "="))
	    (stpos (if start start
		       (forward-word 2) (backward-word 1) (point)))
	    (edpos (set-marker (make-marker)
			       (if end end
				 (max (progn (bpascal-declaration-end)
					     (point))
				      pos))))
	    ind)

	(goto-char stpos)
	;; Indent lines in record block
	(if arg
	    (while (<= (point) (marker-position edpos))
	      (beginning-of-line)
	      (delete-horizontal-space)
	      (if (looking-at "end\\>")
		  (indent-to arg)
		(indent-to (+ arg bpascal-indent-level)))
	      (forward-line 1)))

	;; Do lineup
	(setq ind (bpascal-get-lineup-indent stpos edpos lineup))
	(goto-char stpos)
	(while (<= (point) (marker-position edpos))
	  (if (search-forward lineup (bpascal-get-end-of-line) 'move)
	      (forward-char -1))
	  (delete-horizontal-space)
	  (indent-to ind)
	  (if (not (looking-at lineup))
	      (forward-line 1) ; No more indent if there is no : or =
	    (if (not (looking-at "="))
		(delete-horizontal-space) ;; MK
	      )
	    (forward-char 1)
	    (delete-horizontal-space)
	    (insert " ")
	    ;; Indent record block
	    (if (looking-at "\\(record\\|object\\)\\>")
		(bpascal-indent-declaration (current-column))) ;;;;&&
	    (forward-line 1)))))

    ;; If arg - move point
    (if arg (forward-line -1)
      (goto-char (marker-position pos)))))

;  "Return the indent level that will line up several lines within the region
;from b to e nicely. The lineup string is str."
(defun bpascal-get-lineup-indent (b e str)
  (save-excursion
    (let ((ind 0)
	  (reg (concat str "\\|\\(\\<\\(record\\|object\\)\\>\\)"))
	  nest)
      (goto-char b)
      ;; Get rightmost position
      (while (< (point) e)
	(setq nest 1)
	(if (re-search-forward reg (min e (bpascal-get-end-of-line 2)) 'move)
	    (progn
	      ;; Skip record blocks
	      (if (match-beginning 1)
		  (bpascal-declaration-end)
		(progn
		  (goto-char (match-beginning 0))
		  (skip-chars-backward " \t")
		  (if (> (current-column) ind)
		      (setq ind (current-column)))
		  (goto-char (match-end 0)))))))
      ;; In case no lineup was found
      (if (> ind 0)
	  (1+ ind)
	;; No lineup-string found
	(goto-char b)
	(end-of-line)
	(skip-chars-backward " \t")
	(1+ (current-column))))))
    


;;;
;;; Completion
;;;
(defvar bpascal-str nil)
(defvar bpascal-all nil)
(defvar bpascal-pred nil)
(defvar bpascal-buffer-to-use nil)
(defvar bpascal-flag nil)

(defun bpascal-string-diff (str1 str2)
  "Return index of first letter where STR1 and STR2 differs."
  (catch 'done
    (let ((diff 0))
      (while t
	(if (or (> (1+ diff) (length str1))
		(> (1+ diff) (length str2)))
	    (throw 'done diff))
	(or (equal (aref str1 diff) (aref str2 diff))
	    (throw 'done diff))
	(setq diff (1+ diff))))))

;; Calculate all possible completions for functions if argument is `function',
;; completions for procedures if argument is `procedure' or both functions and
;; procedures otherwise.

(defun bpascal-func-completion (type)
  ;; Build regular expression for function/procedure names
  (if (string= bpascal-str "")
      (setq bpascal-str "[a-zA-Z_]"))
  (let ((bpascal-str (concat (cond
			     ((eq type 'procedure) "\\<\\(procedure\\)\\s +")
			     ((eq type 'function) "\\<\\(function\\)\\s +")
			     (t "\\<\\(function\\constructor\\|destructor\\|procedure\\)\\s +"))
			    "\\<\\(" bpascal-str "[a-zA-Z0-9_.]*\\)\\>"))
	match)
    
    (if (not (looking-at "\\<\\(function\\|destructor\\|constructor\\|procedure\\)\\>"))
	(re-search-backward "\\<\\(function\\|destructor\\|constructor\\|procedure\\)\\>" nil t))
    (forward-char 1)

    ;; Search through all reachable functions
    (while (bpascal-beg-of-defun)
      (if (re-search-forward bpascal-str (bpascal-get-end-of-line) t)
	  (progn (setq match (buffer-substring (match-beginning 2)
					       (match-end 2)))
		 (if (or (null bpascal-pred)
			 (funcall bpascal-pred match))
		     (setq bpascal-all (cons match bpascal-all)))))
      (goto-char (match-beginning 0)))))

(defun bpascal-get-completion-decl ()
  ;; Macro for searching through current declaration (var, type or const)
  ;; for matches of `str' and adding the occurrence tp `all'
  (let ((end (save-excursion (bpascal-declaration-end)
			     (point)))
	match)
    ;; Traverse lines
    (while (< (point) end)
      (if (re-search-forward "[:=]" (bpascal-get-end-of-line) t)
	  ;; Traverse current line
	  (while (and (re-search-backward 
		       (concat "\\((\\|\\<\\(var\\|type\\|const\\)\\>\\)\\|" 
			       bpascal-symbol-re)
		       (bpascal-get-beg-of-line) t)
		      (not (match-end 1)))
	    (setq match (buffer-substring (match-beginning 0) (match-end 0)))
	    (if (string-match (concat "\\<" bpascal-str) match)
		(if (or (null bpascal-pred)
			(funcall bpascal-pred match))
		    (setq bpascal-all (cons match bpascal-all))))))
      (if (re-search-forward "\\<\\(record\\|object\\)\\>" (bpascal-get-end-of-line) t)
	  (bpascal-declaration-end)
	(forward-line 1)))))

(defun bpascal-type-completion ()
  "Calculate all possible completions for types."
  (let ((start (point))
	goon)
    ;; Search for all reachable type declarations
    (while (or (bpascal-beg-of-defun)
	       (setq goon (not goon)))
      (save-excursion
	(if (and (< start (prog1 (save-excursion (bpascal-end-of-defun)
						 (point))
			    (forward-char 1)))
		 (re-search-forward
		  "\\<type\\>\\|\\<\\(begin\\|function\\|destructor\\|constructor\\|procedure\\)\\>"
		  start t)
		 (not (match-end 1)))
	    ;; Check current type declaration
	    (bpascal-get-completion-decl))))))

(defun bpascal-var-completion ()
  "Calculate all possible completions for variables (or constants)."
  (let ((start (point))
	goon twice)
    ;; Search for all reachable var declarations
    (while (or (bpascal-beg-of-defun)
	       (setq goon (not goon)))
      (save-excursion
	(if (> start (prog1 (save-excursion (bpascal-end-of-defun)
					    (point))))
	    () ; Declarations not reachable
	  (if (search-forward "(" (bpascal-get-end-of-line) t)
	      ;; Check parameterlist
		(bpascal-get-completion-decl))
	  (setq twice 2)
	  (while (>= (setq twice (1- twice)) 0)
	    (cond ((and (re-search-forward
			 (concat "\\<\\(var\\|const\\)\\>\\|"
				 "\\<\\(begin\\|function\\|destructor\\|constructor\\|procedure\\)\\>")
			 start t)
			(not (match-end 2)))
		   ;; Check var/const declarations
		   (bpascal-get-completion-decl))
		  ((match-end 2)
		   (setq twice 0)))))))))


(defun bpascal-keyword-completion (keyword-list)
  "Give list of all possible completions of keywords in KEYWORD-LIST."
  (mapcar '(lambda (s) 
	     (if (string-match (concat "\\<" bpascal-str) s)
		 (if (or (null bpascal-pred)
			 (funcall bpascal-pred s))
		     (setq bpascal-all (cons s bpascal-all)))))
	  keyword-list))

;; Function passed to completing-read, try-completion or
;; all-completions to get completion on STR. If predicate is non-nil,
;; it must be a function to be called for every match to check if this
;; should really be a match. If flag is t, the function returns a list
;; of all possible completions. If it is nil it returns a string, the
;; longest possible completion, or t if STR is an exact match. If flag
;; is 'lambda, the function returns t if STR is an exact match, nil
;; otherwise.

(defun bpascal-completion (bpascal-str bpascal-pred bpascal-flag)
  (save-excursion
    (let ((bpascal-all nil))
      ;; Set buffer to use for searching labels. This should be set
      ;; within functions which use bpascal-completions
      (set-buffer bpascal-buffer-to-use)

      ;; Determine what should be completed
      (let ((state (car (bpascal-calculate-indent))))
	(cond (;--Within a declaration or parameterlist
	       (or (eq state 'declaration) (eq state 'paramlist)
		   (and (eq state 'defun)
			(save-excursion
			  (re-search-backward ")[ \t]*:"
					      (bpascal-get-beg-of-line) t))))
	       (if (or (eq state 'paramlist) (eq state 'defun))
		   (bpascal-beg-of-defun))
	       (bpascal-type-completion)
	       (bpascal-keyword-completion bpascal-type-keywords))
	      (;--Starting a new statement
	       (and (not (eq state 'contexp))
		    (save-excursion
		      (skip-chars-backward "a-zA-Z0-9_.")
		      (backward-sexp 1)
		      (or (looking-at bpascal-nosemi-re)
			  (progn
			    (forward-sexp 1)
			    (looking-at "\\s *\\(;\\|:[^=]\\)")))))
	       (save-excursion (bpascal-var-completion))
	       (bpascal-func-completion 'procedure)
	       (bpascal-keyword-completion bpascal-start-keywords))
	      (t;--Anywhere else
	       (save-excursion (bpascal-var-completion))
	       (bpascal-func-completion 'function)
	       (bpascal-keyword-completion bpascal-separator-keywords))))
      
      ;; Now we have built a list of all matches. Give response to caller
      (bpascal-completion-response))))

(defun bpascal-completion-response ()
  (cond ((or (equal bpascal-flag 'lambda) (null bpascal-flag))
	 ;; This was not called by all-completions
	 (if (null bpascal-all)
	     ;; Return nil if there was no matching label
	     nil
	   ;; Get longest string common in the labels
	   (let* ((elm (cdr bpascal-all))
		  (match (car bpascal-all))
		  (min (length match))
		  exact tmp)
	     (if (string= match bpascal-str)
		 ;; Return t if first match was an exact match
		 (setq match t)
	       (while (not (null elm))
		 ;; Find longest common string
		 (if (< (setq tmp (bpascal-string-diff match (car elm))) min)
		     (progn
		       (setq min tmp)
		       (setq match (substring match 0 min))))
		 ;; Terminate with match=t if this is an exact match
		 (if (string= (car elm) bpascal-str)
		     (progn
		       (setq match t)
		       (setq elm nil))
		   (setq elm (cdr elm)))))
	     ;; If this is a test just for exact match, return nil ot t
	     (if (and (equal bpascal-flag 'lambda) (not (equal match 't)))
		 nil
	       match))))
	;; If flag is t, this was called by all-completions. Return
	;; list of all possible completions
	(bpascal-flag
	 bpascal-all)))

(defvar bpascal-last-word-numb 0)
(defvar bpascal-last-word-shown nil)
(defvar bpascal-last-completions nil)

(defun bpascal-complete-word ()
  "Complete word at current point.
\(See also `bpascal-toggle-completions', `bpascal-type-keywords',
`bpascal-start-keywords' and `bpascal-separator-keywords'.)"
  (interactive)
  (let* ((b (save-excursion (skip-chars-backward "a-zA-Z0-9_") (point)))
	 (e (save-excursion (skip-chars-forward "a-zA-Z0-9_") (point)))
	 (bpascal-str (buffer-substring b e))
	 ;; The following variable is used in bpascal-completion
	 (bpascal-buffer-to-use (current-buffer))
	 (allcomp (if (and bpascal-toggle-completions
			   (string= bpascal-last-word-shown bpascal-str))
		      bpascal-last-completions
		    (all-completions bpascal-str 'bpascal-completion)))
	 (match (if bpascal-toggle-completions
		    "" (try-completion
			bpascal-str (mapcar '(lambda (elm)
					      (cons elm 0)) allcomp)))))
    ;; Delete old string
    (delete-region b e)

    ;; Toggle-completions inserts whole labels
    (if bpascal-toggle-completions
	(progn
	  ;; Update entry number in list
	  (setq bpascal-last-completions allcomp
		bpascal-last-word-numb 
		(if (>= bpascal-last-word-numb (1- (length allcomp)))
		    0
		  (1+ bpascal-last-word-numb)))
	  (setq bpascal-last-word-shown (elt allcomp bpascal-last-word-numb))
	  ;; Display next match or same string if no match was found
	  (if (not (null allcomp))
	      (insert "" bpascal-last-word-shown)
	    (insert "" bpascal-str)
	    (message "(No match)")))
      ;; The other form of completion does not necessarily do that.

      ;; Insert match if found, or the original string if no match
      (if (or (null match) (equal match 't))
	  (progn (insert "" bpascal-str)
		 (message "(No match)"))
	(insert "" match))
      ;; Give message about current status of completion
      (cond ((equal match 't)
	     (if (not (null (cdr allcomp)))
		 (message "(Complete but not unique)")
	       (message "(Sole completion)")))
	    ;; Display buffer if the current completion didn't help 
	    ;; on completing the label.
	    ((and (not (null (cdr allcomp))) (= (length bpascal-str)
						(length match)))
	     (with-output-to-temp-buffer "*Completions*"
	       (display-completion-list allcomp))
	     ;; Wait for a keypress. Then delete *Completion*  window
	     (momentary-string-display "" (point))
	     (delete-window (get-buffer-window (get-buffer "*Completions*")))
	     )))))

(defun bpascal-show-completions ()
  "Show all possible completions at current point."
  (interactive)
  (let* ((b (save-excursion (skip-chars-backward "a-zA-Z0-9_") (point)))
	 (e (save-excursion (skip-chars-forward "a-zA-Z0-9_") (point)))
	 (bpascal-str (buffer-substring b e))
	 ;; The following variable is used in bpascal-completion
	 (bpascal-buffer-to-use (current-buffer))
	 (allcomp (if (and bpascal-toggle-completions
			   (string= bpascal-last-word-shown bpascal-str))
		      bpascal-last-completions
		    (all-completions bpascal-str 'bpascal-completion))))
    ;; Show possible completions in a temporary buffer.
    (with-output-to-temp-buffer "*Completions*"
      (display-completion-list allcomp))
    ;; Wait for a keypress. Then delete *Completion*  window
    (momentary-string-display "" (point))
    (delete-window (get-buffer-window (get-buffer "*Completions*")))))


(defun bpascal-get-default-symbol ()
  "Return symbol around current point as a string."
  (save-excursion
    (buffer-substring (progn
			(skip-chars-backward " \t")
			(skip-chars-backward "a-zA-Z0-9_")
			(point))
		      (progn
			(skip-chars-forward "a-zA-Z0-9_")
			(point)))))

(defun bpascal-build-defun-re (str &optional arg)
  "Return function/procedure starting with STR as regular expression.
With optional second arg non-nil, STR is the complete name of the instruction."
  (if arg
      (concat "^\\(function\\|destructor\\|constructor\\|procedure\\)[ \t]+\\(" str "\\)\\>")
    (concat "^\\(function\\|destructor\\|constructor\\|procedure\\)[ \t]+\\(" str "[a-zA-Z0-9_]*\\)\\>")))

;; Function passed to completing-read, try-completion or
;; all-completions to get completion on any function name. If
;; predicate is non-nil, it must be a function to be called for every
;; match to check if this should really be a match. If flag is t, the
;; function returns a list of all possible completions. If it is nil
;; it returns a string, the longest possible completion, or t if STR
;; is an exact match. If flag is 'lambda, the function returns t if
;; STR is an exact match, nil otherwise.

(defun bpascal-comp-defun (bpascal-str bpascal-pred bpascal-flag)
  (save-excursion
    (let ((bpascal-all nil)
	  match)

      ;; Set buffer to use for searching labels. This should be set
      ;; within functions which use bpascal-completions
      (set-buffer bpascal-buffer-to-use)

      (let ((bpascal-str bpascal-str))
	;; Build regular expression for functions
	(if (string= bpascal-str "")
	    (setq bpascal-str (bpascal-build-defun-re "[a-zA-Z_]"))
	  (setq bpascal-str (bpascal-build-defun-re bpascal-str)))
	(goto-char (point-min))
      
	;; Build a list of all possible completions
	(while (re-search-forward bpascal-str nil t)
	  (setq match (buffer-substring (match-beginning 2) (match-end 2)))
	  (if (or (null bpascal-pred)
		  (funcall bpascal-pred match))
	      (setq bpascal-all (cons match bpascal-all)))))

      ;; Now we have built a list of all matches. Give response to caller
      (bpascal-completion-response))))

(defun bpascal-goto-defun ()
  "Move to specified Bpascal function/procedure.
The default is a name found in the buffer around point."
  (interactive)
  (let* ((default (bpascal-get-default-symbol))
	 ;; The following variable is used in bpascal-comp-function
	 (bpascal-buffer-to-use (current-buffer))
	 (default (if (bpascal-comp-defun default nil 'lambda)
		      default ""))
	 (label (if (not (string= default ""))
		    ;; Do completion with default
		    (completing-read (concat "Label: (default " default ") ")
				     'bpascal-comp-defun nil t "")
		  ;; There is no default value. Complete without it
		  (completing-read "Label: "
				   'bpascal-comp-defun nil t ""))))
    ;; If there was no response on prompt, use default value
    (if (string= label "")
	(setq label default))
    ;; Goto right place in buffer if label is not an empty string
    (or (string= label "")
	(progn
	  (goto-char (point-min))
	  (re-search-forward (bpascal-build-defun-re label t))
	  (beginning-of-line)))))



;;;
;;; Bpascal-outline-mode
;;;
(defvar bpascal-outline-map nil "Keymap used in Bpascal Outline mode.")

(if bpascal-outline-map
    nil
  (if (boundp 'set-keymap-name)
      (set-keymap-name bpascal-outline-map 'bpascal-outline-map))
  (if (not (boundp 'set-keymap-parent))
      (setq bpascal-outline-map (copy-keymap bpascal-mode-map))
    (setq bpascal-outline-map (make-sparse-keymap))
    (set-keymap-parent bpascal-outline-map bpascal-mode-map))
  (define-key bpascal-outline-map "\M-\C-a"  'bpascal-outline-prev-defun)
  (define-key bpascal-outline-map "\M-\C-e"  'bpascal-outline-next-defun)
  (define-key bpascal-outline-map "\C-c\C-d" 'bpascal-outline-goto-defun)
  (define-key bpascal-outline-map "\C-c\C-s" 'bpascal-show-all)
  (define-key bpascal-outline-map "\C-c\C-h" 'bpascal-hide-other-defuns))

(defvar bpascal-outline-mode nil "Non-nil while using Bpascal Outline mode.")
(make-variable-buffer-local 'bpascal-outline-mode)
(set-default 'bpascal-outline-mode nil)
(if (not (assoc 'bpascal-outline-mode minor-mode-alist))
    (setq minor-mode-alist (append minor-mode-alist
				   (list '(bpascal-outline-mode " Outl")))))

(defun bpascal-outline (&optional arg)
  "Outline-line minor mode for Bpascal mode.
When in Bpascal Outline mode, portions
of the text being edited may be made invisible. \\<bpascal-outline-map>

Bpascal Outline mode provides some additional commands.

\\[bpascal-outline-prev-defun]\
\t- Move to previous function/procedure, hiding everything else.
\\[bpascal-outline-next-defun]\
\t- Move to next function/procedure, hiding everything else.
\\[bpascal-outline-goto-defun]\
\t- Goto function/procedure prompted for in minibuffer,
\t  hide all other functions.
\\[bpascal-show-all]\t- Show the whole buffer.
\\[bpascal-hide-other-defuns]\
\t- Hide everything but the current function (function under the cursor).
\\[bpascal-outline]\t- Leave bpascal-outline-mode."
  (interactive "P")
  (setq bpascal-outline-mode
	(if (null arg) (not bpascal-outline-mode) t))
  (if (boundp 'redraw-mode-line)
      (redraw-mode-line))
  (if bpascal-outline-mode
      (progn
	(setq selective-display t)
	(use-local-map bpascal-outline-map))
    (progn
      (setq selective-display nil)
      (bpascal-show-all)
      (use-local-map bpascal-mode-map))))

(defun bpascal-outline-change (b e bpascal-flag)
  (let ((modp (buffer-modified-p)))
    (unwind-protect
	(subst-char-in-region b e (if (= bpascal-flag ?\n) 
				      ?\^M ?\n) bpascal-flag)
      (set-buffer-modified-p modp))))

(defun bpascal-show-all ()
  "Show all of the text in the buffer."
  (interactive)
  (bpascal-outline-change (point-min) (point-max) ?\n))

(defun bpascal-hide-other-defuns ()
  "Show only the current defun."
  (interactive)
  (save-excursion
    (let ((beg (progn (if (not (looking-at "\\(function\\|destructor\\|constructor\\|procedure\\)\\>"))
			  (bpascal-beg-of-defun))
		      (point)))
	  (end (progn (bpascal-end-of-defun)
		      (backward-sexp 1)
		      (search-forward "\n\\|\^M" nil t)
		      (point)))
	  (opoint (point-min)))
      (goto-char (point-min))

      ;; Hide all functions before current function
      (while (re-search-forward "^\\(function\\|destructor\\|constructor\\|procedure\\)\\>" beg 'move)
	(bpascal-outline-change opoint (1- (match-beginning 0)) ?\^M)
	(setq opoint (point))
	;; Functions may be nested
	(if (> (progn (bpascal-end-of-defun) (point)) beg)
	    (goto-char opoint)))
      (if (> beg opoint)
	  (bpascal-outline-change opoint (1- beg) ?\^M))

      ;; Show current function
      (bpascal-outline-change beg end ?\n)
      ;; Hide nested functions
      (forward-char 1)
      (while (re-search-forward "^\\(function\\|destructor\\|constructor\\|procedure\\)\\>" end 'move)
	(setq opoint (point))
	(bpascal-end-of-defun)
	(bpascal-outline-change opoint (point) ?\^M))

      (goto-char end)
      (setq opoint end)

      ;; Hide all function after current function
      (while (re-search-forward "^\\(function\\|destructor\\|constructor\\|procedure\\)\\>" nil 'move)
	(bpascal-outline-change opoint (1- (match-beginning 0)) ?\^M)
	(setq opoint (point))
	(bpascal-end-of-defun))
      (bpascal-outline-change opoint (point-max) ?\^M)

      ;; Hide main program
      (if (< (progn (forward-line -1) (point)) end)
	  (progn
	    (goto-char beg)
	    (bpascal-end-of-defun)
	    (backward-sexp 1)
	    (bpascal-outline-change (point) (point-max) ?\^M))))))

(defun bpascal-outline-next-defun ()
  "Move to next function/procedure, hiding all others."
  (interactive)
  (bpascal-end-of-defun)
  (bpascal-hide-other-defuns))

(defun bpascal-outline-prev-defun ()
  "Move to previous function/procedure, hiding all others."
  (interactive)
  (bpascal-beg-of-defun)
  (bpascal-hide-other-defuns))

(defun bpascal-outline-goto-defun ()
  "Move to specified function/procedure, hiding all others."
  (interactive)
  (bpascal-goto-defun)
  (bpascal-hide-other-defuns))

(defvar auto-mode-alist
  (cons '("\\.pas\\'" . pascal-mode)
	auto-mode-alist))

(defun bpascal-make-imenu ()
  "Do `imenu-add-to-menubar' for this buffer."
  (interactive)
  (imenu-add-to-menubar "Constructs"))

(provide 'bpascal)

;;; bpascal.el ends here

(setq bpascal-indent-level       2)
(setq bpascal-auto-endcomments   t)
